#ifndef HELPERS_H
#define HELPERS_H

#include <arpa/inet.h>
#include <errno.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <linux/netfilter.h>
#include <linux/netfilter/nf_tables.h>
#include <linux/netfilter/nfnetlink.h>
#include <linux/netfilter/nfnetlink_queue.h>

#include <libmnl/libmnl.h>
#include <libnftnl/chain.h>
#include <libnftnl/expr.h>
#include <libnftnl/rule.h>
#include <libnftnl/table.h>

#define INFO(fmt, ...) fprintf(stderr, "[*] " fmt "\n", ##__VA_ARGS__)
#define WARN(fmt, ...) fprintf(stderr, "[!] " fmt "\n", ##__VA_ARGS__)
#define ERROR(msg) fprintf(stderr, "[-] %s:%d: %s: %s\n", __func__, __LINE__, msg, strerror(errno))
#define ARRAY_SIZE(x) (sizeof(x) / sizeof((x)[0]))

struct nft_userdata {
    uint8_t len;
    unsigned char data[];
};

typedef struct mnl_socket *sock;
typedef struct mnl_nlmsg_batch *batch;
typedef struct nlmsghdr *nlmsghdr;

typedef struct nftnl_table *table;
typedef struct nftnl_chain *chain;
typedef struct nftnl_rule *rule;
typedef struct nftnl_expr *expr;

static uint32_t seq = 1, rseq = 1;

batch batch_init(size_t size);
int batch_send_and_run_callbacks(batch b, sock s, void *cb);

table make_table(const char *name, uint32_t family, const void *udata, uint32_t udlen);
void batch_new_table(batch b, table t, uint32_t family);

chain make_chain(const char *table, const char *name, uint32_t flags, uint32_t hooknum, uint32_t prio, char *type);
void batch_new_chain(batch b, chain c, uint32_t family);

rule make_rule(const char *table, const char *chain, expr *exprs, size_t num_exprs, const void *udata, uint32_t udlen, uint64_t handle);
void batch_new_rule(batch b, rule r, uint32_t family);
void batch_del_rule(batch b, rule r, uint32_t family);

expr make_ct_set_zone_expr(uint32_t sreg);
expr make_queue_expr(uint16_t num, uint16_t total, uint16_t flags);
expr make_notrack_expr();
expr make_cmp_expr(uint32_t sreg, uint32_t cmp_op, uint8_t data);
expr make_log_expr(char *prefix);
expr make_payload_expr(uint32_t base, uint32_t offset, uint32_t len, uint32_t dreg);

int run_callbacks(sock s, mnl_cb_t cb, void *data);

nlmsghdr dump_rule(rule r, char *buf, uint32_t family);

#endif // HELPERS_H