/*
 * Decompiled with CFR 0.152.
 */
package io.openmessaging.storage.dledger.store.file;

import io.openmessaging.storage.dledger.store.file.DefaultMmapFile;
import io.openmessaging.storage.dledger.store.file.MmapFile;
import io.openmessaging.storage.dledger.store.file.ReferenceResource;
import io.openmessaging.storage.dledger.store.file.SelectMmapBufferResult;
import io.openmessaging.storage.dledger.utils.DLedgerUtils;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MmapFileList {
    public static final int MIN_BLANK_LEN = 8;
    public static final int BLANK_MAGIC_CODE = -1;
    private static Logger logger = LoggerFactory.getLogger(MmapFile.class);
    private static final int DELETE_FILES_BATCH_MAX = 10;
    private final String storePath;
    private final int mappedFileSize;
    private final CopyOnWriteArrayList<MmapFile> mappedFiles = new CopyOnWriteArrayList();
    private long flushedWhere = 0L;
    private long committedWhere = 0L;
    private volatile long storeTimestamp = 0L;

    public MmapFileList(String storePath, int mappedFileSize) {
        this.storePath = storePath;
        this.mappedFileSize = mappedFileSize;
    }

    public boolean checkSelf() {
        if (!this.mappedFiles.isEmpty()) {
            Iterator<MmapFile> iterator = this.mappedFiles.iterator();
            MmapFile pre = null;
            while (iterator.hasNext()) {
                MmapFile cur = iterator.next();
                if (pre != null && cur.getFileFromOffset() - pre.getFileFromOffset() != (long)this.mappedFileSize) {
                    logger.error("[BUG]The mappedFile queue's data is damaged, the adjacent mappedFile's offset don't match pre file {}, cur file {}", (Object)pre.getFileName(), (Object)cur.getFileName());
                    return false;
                }
                pre = cur;
            }
        }
        return true;
    }

    public MmapFile getMappedFileByTime(long timestamp) {
        Object[] mfs = this.copyMappedFiles();
        if (null == mfs) {
            return null;
        }
        for (int i = 0; i < mfs.length; ++i) {
            MmapFile mappedFile = (MmapFile)mfs[i];
            if (mappedFile.getLastModifiedTimestamp() < timestamp) continue;
            return mappedFile;
        }
        return (MmapFile)mfs[mfs.length - 1];
    }

    private Object[] copyMappedFiles() {
        if (this.mappedFiles.size() <= 0) {
            return null;
        }
        return this.mappedFiles.toArray();
    }

    public void truncateOffset(long offset) {
        Object[] mfs = this.copyMappedFiles();
        if (mfs == null) {
            return;
        }
        ArrayList<MmapFile> willRemoveFiles = new ArrayList<MmapFile>();
        for (int i = 0; i < mfs.length; ++i) {
            MmapFile file = (MmapFile)mfs[i];
            long fileTailOffset = file.getFileFromOffset() + (long)this.mappedFileSize;
            if (fileTailOffset <= offset) continue;
            if (offset >= file.getFileFromOffset()) {
                file.setWrotePosition((int)(offset % (long)this.mappedFileSize));
                file.setCommittedPosition((int)(offset % (long)this.mappedFileSize));
                file.setFlushedPosition((int)(offset % (long)this.mappedFileSize));
                continue;
            }
            willRemoveFiles.add(file);
        }
        this.destroyExpiredFiles(willRemoveFiles);
        this.deleteExpiredFiles(willRemoveFiles);
    }

    void destroyExpiredFiles(List<MmapFile> files) {
        Collections.sort(files, new Comparator<MmapFile>(){

            @Override
            public int compare(MmapFile o1, MmapFile o2) {
                if (o1.getFileFromOffset() < o2.getFileFromOffset()) {
                    return -1;
                }
                if (o1.getFileFromOffset() > o2.getFileFromOffset()) {
                    return 1;
                }
                return 0;
            }
        });
        for (int i = 0; i < files.size(); ++i) {
            MmapFile mmapFile = files.get(i);
            while (!mmapFile.destroy(10000L)) {
                DLedgerUtils.sleep(1000L);
            }
        }
    }

    public void resetOffset(long offset) {
        Object[] mfs = this.copyMappedFiles();
        if (mfs == null) {
            return;
        }
        ArrayList<MmapFile> willRemoveFiles = new ArrayList<MmapFile>();
        for (int i = mfs.length - 1; i >= 0; --i) {
            MmapFile file = (MmapFile)mfs[i];
            long fileTailOffset = file.getFileFromOffset() + (long)this.mappedFileSize;
            if (file.getFileFromOffset() > offset) continue;
            if (offset < fileTailOffset) {
                file.setStartPosition((int)(offset % (long)this.mappedFileSize));
                continue;
            }
            willRemoveFiles.add(file);
        }
        this.destroyExpiredFiles(willRemoveFiles);
        this.deleteExpiredFiles(willRemoveFiles);
    }

    public void updateWherePosition(long wherePosition) {
        if (wherePosition > this.getMaxWrotePosition()) {
            logger.warn("[UpdateWherePosition] wherePosition {} > maxWrotePosition {}", (Object)wherePosition, (Object)this.getMaxWrotePosition());
            return;
        }
        this.setFlushedWhere(wherePosition);
        this.setCommittedWhere(wherePosition);
    }

    public long append(byte[] data) {
        return this.append(data, 0, data.length);
    }

    public long append(byte[] data, int pos, int len) {
        return this.append(data, pos, len, true);
    }

    public long append(byte[] data, boolean useBlank) {
        return this.append(data, 0, data.length, useBlank);
    }

    public long preAppend(int len) {
        return this.preAppend(len, true);
    }

    public long preAppend(int len, boolean useBlank) {
        int blank;
        MmapFile mappedFile = this.getLastMappedFile();
        if (null == mappedFile || mappedFile.isFull()) {
            mappedFile = this.getLastMappedFile(0L);
        }
        if (null == mappedFile) {
            logger.error("Create mapped file for {}", (Object)this.storePath);
            return -1L;
        }
        int n = blank = useBlank ? 8 : 0;
        if (len + blank > mappedFile.getFileSize() - mappedFile.getWrotePosition()) {
            if (blank < 8) {
                logger.error("Blank {} should ge {}", (Object)blank, (Object)8);
                return -1L;
            }
            ByteBuffer byteBuffer = ByteBuffer.allocate(mappedFile.getFileSize() - mappedFile.getWrotePosition());
            byteBuffer.putInt(-1);
            byteBuffer.putInt(mappedFile.getFileSize() - mappedFile.getWrotePosition());
            if (!mappedFile.appendMessage(byteBuffer.array())) {
                logger.error("Append blank error for {}", (Object)this.storePath);
                return -1L;
            }
            mappedFile.setWrotePosition(mappedFile.getFileSize());
            mappedFile = this.getLastMappedFile(0L);
            if (null == mappedFile) {
                logger.error("Create mapped file for {}", (Object)this.storePath);
                return -1L;
            }
        }
        return mappedFile.getFileFromOffset() + (long)mappedFile.getWrotePosition();
    }

    public long append(byte[] data, int pos, int len, boolean useBlank) {
        if (this.preAppend(len, useBlank) == -1L) {
            return -1L;
        }
        MmapFile mappedFile = this.getLastMappedFile();
        long currPosition = mappedFile.getFileFromOffset() + (long)mappedFile.getWrotePosition();
        if (!mappedFile.appendMessage(data, pos, len)) {
            logger.error("Append error for {}", (Object)this.storePath);
            return -1L;
        }
        return currPosition;
    }

    public SelectMmapBufferResult getData(long offset, int size) {
        MmapFile mappedFile = this.findMappedFileByOffset(offset, offset == 0L);
        if (mappedFile != null) {
            int pos = (int)(offset % (long)this.mappedFileSize);
            return mappedFile.selectMappedBuffer(pos, size);
        }
        return null;
    }

    public SelectMmapBufferResult getData(long offset) {
        MmapFile mappedFile = this.findMappedFileByOffset(offset, offset == 0L);
        if (mappedFile != null) {
            int pos = (int)(offset % (long)this.mappedFileSize);
            return mappedFile.selectMappedBuffer(pos);
        }
        return null;
    }

    void deleteExpiredFiles(List<MmapFile> files) {
        if (!files.isEmpty()) {
            Iterator<MmapFile> iterator = files.iterator();
            while (iterator.hasNext()) {
                MmapFile cur = iterator.next();
                if (this.mappedFiles.contains(cur)) continue;
                iterator.remove();
                logger.info("This mappedFile {} is not contained by mappedFiles, so skip it.", (Object)cur.getFileName());
            }
            try {
                if (!this.mappedFiles.removeAll(files)) {
                    logger.error("deleteExpiredFiles remove failed.");
                }
            }
            catch (Exception e) {
                logger.error("deleteExpiredFiles has exception.", e);
            }
        }
    }

    public boolean load() {
        File dir = new File(this.storePath);
        Object[] files = dir.listFiles();
        if (files != null) {
            Arrays.sort(files);
            for (Object file : files) {
                if (((File)file).length() != (long)this.mappedFileSize) {
                    logger.warn(file + "\t" + ((File)file).length() + " length not matched message store config value, please check it manually. You should delete old files before changing mapped file size");
                    return false;
                }
                try {
                    DefaultMmapFile mappedFile = new DefaultMmapFile(((File)file).getPath(), this.mappedFileSize);
                    mappedFile.setWrotePosition(this.mappedFileSize);
                    mappedFile.setFlushedPosition(this.mappedFileSize);
                    mappedFile.setCommittedPosition(this.mappedFileSize);
                    this.mappedFiles.add(mappedFile);
                    logger.info("load " + ((File)file).getPath() + " OK");
                }
                catch (IOException e) {
                    logger.error("load file " + file + " error", e);
                    return false;
                }
            }
        }
        return true;
    }

    public MmapFile getLastMappedFile(long startOffset, boolean needCreate) {
        long createOffset = -1L;
        MmapFile mappedFileLast = this.getLastMappedFile();
        if (mappedFileLast == null) {
            createOffset = startOffset - startOffset % (long)this.mappedFileSize;
        } else if (mappedFileLast.isFull()) {
            createOffset = mappedFileLast.getFileFromOffset() + (long)this.mappedFileSize;
        }
        if (createOffset != -1L && needCreate) {
            String nextFilePath = this.storePath + File.separator + DLedgerUtils.offset2FileName(createOffset);
            DefaultMmapFile mappedFile = null;
            try {
                mappedFile = new DefaultMmapFile(nextFilePath, this.mappedFileSize);
            }
            catch (IOException e) {
                logger.error("create mappedFile exception", e);
            }
            if (mappedFile != null) {
                if (this.mappedFiles.isEmpty()) {
                    mappedFile.setFirstCreateInQueue(true);
                }
                this.mappedFiles.add(mappedFile);
            }
            return mappedFile;
        }
        return mappedFileLast;
    }

    public MmapFile getLastMappedFile(long startOffset) {
        return this.getLastMappedFile(startOffset, true);
    }

    public MmapFile getLastMappedFile() {
        MmapFile mappedFileLast = null;
        while (!this.mappedFiles.isEmpty()) {
            try {
                mappedFileLast = this.mappedFiles.get(this.mappedFiles.size() - 1);
                break;
            }
            catch (IndexOutOfBoundsException indexOutOfBoundsException) {
            }
            catch (Exception e) {
                logger.error("getLastMappedFile has exception.", e);
                break;
            }
        }
        return mappedFileLast;
    }

    public long getMinOffset() {
        MmapFile mmapFile = this.getFirstMappedFile();
        if (mmapFile != null) {
            return mmapFile.getFileFromOffset() + (long)mmapFile.getStartPosition();
        }
        return 0L;
    }

    public long getMaxReadPosition() {
        MmapFile mappedFile = this.getLastMappedFile();
        if (mappedFile != null) {
            return mappedFile.getFileFromOffset() + (long)mappedFile.getReadPosition();
        }
        return 0L;
    }

    public long getMaxWrotePosition() {
        MmapFile mappedFile = this.getLastMappedFile();
        if (mappedFile != null) {
            return mappedFile.getFileFromOffset() + (long)mappedFile.getWrotePosition();
        }
        return 0L;
    }

    public long remainHowManyDataToCommit() {
        return this.getMaxWrotePosition() - this.committedWhere;
    }

    public long remainHowManyDataToFlush() {
        return this.getMaxReadPosition() - this.flushedWhere;
    }

    public void deleteLastMappedFile() {
        MmapFile lastMappedFile = this.getLastMappedFile();
        if (lastMappedFile != null) {
            lastMappedFile.destroy(1000L);
            this.mappedFiles.remove(lastMappedFile);
            logger.info("on recover, destroy a logic mapped file " + lastMappedFile.getFileName());
        }
    }

    public int deleteExpiredFileByTime(long expiredTime, int deleteFilesInterval, long intervalForcibly, boolean cleanImmediately) {
        Object[] mfs = this.copyMappedFiles();
        if (null == mfs) {
            return 0;
        }
        int mfsLength = mfs.length - 1;
        int deleteCount = 0;
        ArrayList<MmapFile> files = new ArrayList<MmapFile>();
        if (null != mfs) {
            for (int i = 0; i < mfsLength; ++i) {
                MmapFile mappedFile = (MmapFile)mfs[i];
                long liveMaxTimestamp = mappedFile.getLastModifiedTimestamp() + expiredTime;
                if (System.currentTimeMillis() < liveMaxTimestamp && !cleanImmediately || !mappedFile.destroy(intervalForcibly)) break;
                files.add(mappedFile);
                ++deleteCount;
                if (files.size() >= 10) break;
                if (deleteFilesInterval <= 0 || i + 1 >= mfsLength) continue;
                try {
                    Thread.sleep(deleteFilesInterval);
                    continue;
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
        }
        this.deleteExpiredFiles(files);
        return deleteCount;
    }

    public int deleteExpiredFileByOffset(long offset, int unitSize) {
        Object[] mfs = this.copyMappedFiles();
        ArrayList<MmapFile> files = new ArrayList<MmapFile>();
        int deleteCount = 0;
        if (null != mfs) {
            int mfsLength = mfs.length - 1;
            for (int i = 0; i < mfsLength; ++i) {
                boolean destroy;
                MmapFile mappedFile = (MmapFile)mfs[i];
                SelectMmapBufferResult result = mappedFile.selectMappedBuffer(this.mappedFileSize - unitSize);
                if (result != null) {
                    long maxOffsetInLogicQueue = result.getByteBuffer().getLong();
                    result.release();
                    boolean bl = destroy = maxOffsetInLogicQueue < offset;
                    if (destroy) {
                        logger.info("physic min offset " + offset + ", logics in current mappedFile max offset " + maxOffsetInLogicQueue + ", delete it");
                    }
                } else if (!mappedFile.isAvailable()) {
                    logger.warn("Found a hanged consume queue file, attempting to delete it.");
                    destroy = true;
                } else {
                    logger.warn("this being not executed forever.");
                    break;
                }
                if (!destroy || !mappedFile.destroy(60000L)) break;
                files.add(mappedFile);
                ++deleteCount;
            }
        }
        this.deleteExpiredFiles(files);
        return deleteCount;
    }

    public boolean flush(int flushLeastPages) {
        boolean result = true;
        MmapFile mappedFile = this.findMappedFileByOffset(this.flushedWhere, this.flushedWhere == 0L);
        if (mappedFile != null) {
            int offset = mappedFile.flush(flushLeastPages);
            long where = mappedFile.getFileFromOffset() + (long)offset;
            result = where == this.flushedWhere;
            this.flushedWhere = where;
        }
        return result;
    }

    public boolean commit(int commitLeastPages) {
        boolean result = true;
        MmapFile mappedFile = this.findMappedFileByOffset(this.committedWhere, this.committedWhere == 0L);
        if (mappedFile != null) {
            int offset = mappedFile.commit(commitLeastPages);
            long where = mappedFile.getFileFromOffset() + (long)offset;
            result = where == this.committedWhere;
            this.committedWhere = where;
        }
        return result;
    }

    public MmapFile findMappedFileByOffset(long offset, boolean returnFirstOnNotFound) {
        block9: {
            try {
                MmapFile firstMappedFile = this.getFirstMappedFile();
                MmapFile lastMappedFile = this.getLastMappedFile();
                if (firstMappedFile == null || lastMappedFile == null) break block9;
                if (offset < firstMappedFile.getFileFromOffset() || offset >= lastMappedFile.getFileFromOffset() + (long)this.mappedFileSize) {
                    logger.warn("Offset not matched. Request offset: {}, firstOffset: {}, lastOffset: {}, mappedFileSize: {}, mappedFiles count: {}", offset, firstMappedFile.getFileFromOffset(), lastMappedFile.getFileFromOffset() + (long)this.mappedFileSize, this.mappedFileSize, this.mappedFiles.size());
                } else {
                    int index = (int)(offset / (long)this.mappedFileSize - firstMappedFile.getFileFromOffset() / (long)this.mappedFileSize);
                    MmapFile targetFile = null;
                    try {
                        targetFile = this.mappedFiles.get(index);
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                    if (targetFile != null && offset >= targetFile.getFileFromOffset() && offset < targetFile.getFileFromOffset() + (long)this.mappedFileSize) {
                        return targetFile;
                    }
                    logger.warn("Offset is matched, but get file failed, maybe the file number is changed. Request offset: {}, firstOffset: {}, lastOffset: {}, mappedFileSize: {}, mappedFiles count: {}", offset, firstMappedFile.getFileFromOffset(), lastMappedFile.getFileFromOffset() + (long)this.mappedFileSize, this.mappedFileSize, this.mappedFiles.size());
                    for (MmapFile tmpMappedFile : this.mappedFiles) {
                        if (offset < tmpMappedFile.getFileFromOffset() || offset >= tmpMappedFile.getFileFromOffset() + (long)this.mappedFileSize) continue;
                        return tmpMappedFile;
                    }
                }
                if (returnFirstOnNotFound) {
                    return firstMappedFile;
                }
            }
            catch (Exception e) {
                logger.error("findMappedFileByOffset Exception", e);
            }
        }
        return null;
    }

    public MmapFile getFirstMappedFile() {
        MmapFile mappedFileFirst = null;
        if (!this.mappedFiles.isEmpty()) {
            try {
                mappedFileFirst = this.mappedFiles.get(0);
            }
            catch (IndexOutOfBoundsException indexOutOfBoundsException) {
            }
            catch (Exception e) {
                logger.error("getFirstMappedFile has exception.", e);
            }
        }
        return mappedFileFirst;
    }

    public MmapFile findMappedFileByOffset(long offset) {
        return this.findMappedFileByOffset(offset, false);
    }

    public long getMappedMemorySize() {
        long size = 0L;
        Object[] mfs = this.copyMappedFiles();
        if (mfs != null) {
            for (Object mf : mfs) {
                if (!((ReferenceResource)mf).isAvailable()) continue;
                size += (long)this.mappedFileSize;
            }
        }
        return size;
    }

    public boolean retryDeleteFirstFile(long intervalForcibly) {
        MmapFile mappedFile = this.getFirstMappedFile();
        if (mappedFile != null && !mappedFile.isAvailable()) {
            logger.warn("the mappedFile was destroyed once, but still alive, " + mappedFile.getFileName());
            boolean result = mappedFile.destroy(intervalForcibly);
            if (result) {
                logger.info("the mappedFile re delete OK, " + mappedFile.getFileName());
                ArrayList<MmapFile> tmpFiles = new ArrayList<MmapFile>();
                tmpFiles.add(mappedFile);
                this.deleteExpiredFiles(tmpFiles);
            } else {
                logger.warn("the mappedFile re delete failed, " + mappedFile.getFileName());
            }
            return result;
        }
        return false;
    }

    public void shutdown(long intervalForcibly) {
        for (MmapFile mf : this.mappedFiles) {
            mf.shutdown(intervalForcibly);
        }
    }

    public void destroy() {
        for (MmapFile mf : this.mappedFiles) {
            mf.destroy(3000L);
        }
        this.mappedFiles.clear();
        this.flushedWhere = 0L;
        File file = new File(this.storePath);
        if (file.isDirectory()) {
            file.delete();
        }
    }

    public boolean rebuildWithPos(long pos) {
        this.truncateOffset(-1L);
        this.getLastMappedFile(pos);
        this.truncateOffset(pos);
        this.resetOffset(pos);
        return pos == this.getMaxWrotePosition() && pos == this.getMinOffset();
    }

    public long getFlushedWhere() {
        return this.flushedWhere;
    }

    public void setFlushedWhere(long flushedWhere) {
        this.flushedWhere = flushedWhere;
    }

    public long getStoreTimestamp() {
        return this.storeTimestamp;
    }

    public List<MmapFile> getMappedFiles() {
        return this.mappedFiles;
    }

    public int getMappedFileSize() {
        return this.mappedFileSize;
    }

    public long getCommittedWhere() {
        return this.committedWhere;
    }

    public void setCommittedWhere(long committedWhere) {
        this.committedWhere = committedWhere;
    }
}

