#pragma once
#ifndef VMMDEV_H
#define VMMDEV_H
#include<linux/types.h>
/** @name HGCM flags.
 * @{
 */
# define VBOX_HGCM_REQ_DONE      1
# define VBOX_HGCM_REQ_CANCELLED (0x2)
/** @} */

/** @name Host-Guest Communication Manager (HGCM) Status Codes
 * @{
 */
/** Requested service does not exist. */
#define VERR_HGCM_SERVICE_NOT_FOUND                 (-2900)
/** Service rejected client connection */
#define VINF_HGCM_CLIENT_REJECTED                   2901
/** Command address is invalid. */
#define VERR_HGCM_INVALID_CMD_ADDRESS               (-2902)
/** Service will execute the command in background. */
#define VINF_HGCM_ASYNC_EXECUTE                     2903
/** HGCM could not perform requested operation because of an internal error. */
#define VERR_HGCM_INTERNAL                          (-2904)
/** Invalid HGCM client id. */
#define VERR_HGCM_INVALID_CLIENT_ID                 (-2905)
/** The HGCM is saving state. */
#define VINF_HGCM_SAVE_STATE                        (2906)
/** Requested service already exists. */
#define VERR_HGCM_SERVICE_EXISTS                    (-2907)
/** Too many clients for the service. */
#define VERR_HGCM_TOO_MANY_CLIENTS                  (-2908)
/** Too many calls to the service from a client. */
#define VERR_HGCM_TOO_MANY_CLIENT_CALLS             (-2909)
/** @} */


/** Guest Physical Memory Address; limited to 32 bits.*/
typedef uint32_t                RTGCPHYS32;
/** Pointer to Guest Physical Memory Address. */
typedef RTGCPHYS32       *PRTGCPHYS32;
/** Pointer to const Guest Physical Memory Address. */
typedef const RTGCPHYS32 *PCRTGCPHYS32;
/** @def NIL_RTGCPHYS32
 * NIL GC Physical Address.
 * NIL_RTGCPHYS32 is used to signal an invalid physical address, similar
 * to the NULL pointer. Note that this value may actually be valid in
 * some contexts.
 */
#define NIL_RTGCPHYS32          (~(RTGCPHYS32)0)


/** Guest Physical Memory Address; limited to 64 bits.*/
typedef uint64_t                RTGCPHYS64;
/** Pointer to Guest Physical Memory Address. */
typedef RTGCPHYS64      *PRTGCPHYS64;
/** Pointer to const Guest Physical Memory Address. */
typedef const RTGCPHYS64 *PCRTGCPHYS64;
/** @def NIL_RTGCPHYS64
 * NIL GC Physical Address.
 * NIL_RTGCPHYS64 is used to signal an invalid physical address, similar
 * to the NULL pointer. Note that this value may actually be valid in
 * some contexts.
 */
#define NIL_RTGCPHYS64          (~(RTGCPHYS64)0)

typedef uint32_t RTGCPTR32;
typedef uint64_t RTGCPTR64;

/**
 * HGCM service location types.
 * @ingroup grp_vmmdev_req
 */
typedef enum
{
    VMMDevHGCMLoc_Invalid    = 0,
    VMMDevHGCMLoc_LocalHost  = 1,
    VMMDevHGCMLoc_LocalHost_Existing = 2,
    VMMDevHGCMLoc_SizeHack   = 0x7fffffff
} HGCMServiceLocationType;
//AssertCompileSize(HGCMServiceLocationType, 4);

/**
 * HGCM host service location.
 * @ingroup grp_vmmdev_req
 */
typedef struct
{
    char achName[128]; /**< This is really szName. */
} HGCMServiceLocationHost;
//AssertCompileSize(HGCMServiceLocationHost, 128);

/**
 * HGCM service location.
 * @ingroup grp_vmmdev_req
 */
typedef struct HGCMSERVICELOCATION
{
    /** Type of the location. */
    HGCMServiceLocationType type;

    union
    {
        HGCMServiceLocationHost host;
    } u;
} HGCMServiceLocation;
//AssertCompileSize(HGCMServiceLocation, 128+4);

/**
 * VMMDev request types.
 * @note when updating this, adjust vmmdevGetRequestSize() as well
 */
typedef enum VMMDevRequestType
{
    VMMDevReq_InvalidRequest             =  0,
    VMMDevReq_GetMouseStatus             =  1,
    VMMDevReq_SetMouseStatus             =  2,
    VMMDevReq_SetPointerShape            =  3,
    VMMDevReq_GetHostVersion             =  4,
    VMMDevReq_Idle                       =  5,
    VMMDevReq_GetHostTime                = 10,
    VMMDevReq_GetHypervisorInfo          = 20,
    VMMDevReq_SetHypervisorInfo          = 21,
    VMMDevReq_RegisterPatchMemory        = 22, /**< @since version 3.0.6 */
    VMMDevReq_DeregisterPatchMemory      = 23, /**< @since version 3.0.6 */
    VMMDevReq_SetPowerStatus             = 30,
    VMMDevReq_AcknowledgeEvents          = 41,
    VMMDevReq_CtlGuestFilterMask         = 42,
    VMMDevReq_ReportGuestInfo            = 50,
    VMMDevReq_ReportGuestInfo2           = 58, /**< @since version 3.2.0 */
    VMMDevReq_ReportGuestStatus          = 59, /**< @since version 3.2.8 */
    VMMDevReq_ReportGuestUserState       = 74, /**< @since version 4.3 */
    /**
     * Retrieve a display resize request sent by the host using
     * @a IDisplay:setVideoModeHint.  Deprecated.
     *
     * Similar to @a VMMDevReq_GetDisplayChangeRequest2, except that it only
     * considers host requests sent for the first virtual display.  This guest
     * request should not be used in new guest code, and the results are
     * undefined if a guest mixes calls to this and
     * @a VMMDevReq_GetDisplayChangeRequest2.
     */
    VMMDevReq_GetDisplayChangeRequest    = 51,
    VMMDevReq_VideoModeSupported         = 52,
    VMMDevReq_GetHeightReduction         = 53,
    /**
     * Retrieve a display resize request sent by the host using
     * @a IDisplay:setVideoModeHint.
     *
     * Queries a display resize request sent from the host.  If the
     * @a eventAck member is sent to true and there is an unqueried
     * request available for one of the virtual display then that request will
     * be returned.  If several displays have unqueried requests the lowest
     * numbered display will be chosen first.  Only the most recent unseen
     * request for each display is remembered.
     * If @a eventAck is set to false, the last host request queried with
     * @a eventAck set is resent, or failing that the most recent received from
     * the host.  If no host request was ever received then all zeros are
     * returned.
     */
    VMMDevReq_GetDisplayChangeRequest2   = 54,
    VMMDevReq_ReportGuestCapabilities    = 55,
    VMMDevReq_SetGuestCapabilities       = 56,
    VMMDevReq_VideoModeSupported2        = 57, /**< @since version 3.2.0 */
    VMMDevReq_GetDisplayChangeRequestEx  = 80, /**< @since version 4.2.4 */
    VMMDevReq_GetDisplayChangeRequestMulti = 81,
    VMMDevReq_HGCMConnect                = 60,
    VMMDevReq_HGCMDisconnect             = 61,
    VMMDevReq_HGCMCall32                 = 62,
    VMMDevReq_HGCMCall64                 = 63,
    VMMDevReq_HGCMCall                   = VMMDevReq_HGCMCall64,
    VMMDevReq_HGCMCancel                 = 64,
    VMMDevReq_HGCMCancel2                = 65,
    VMMDevReq_VideoAccelEnable           = 70,
    VMMDevReq_VideoAccelFlush            = 71,
    VMMDevReq_VideoSetVisibleRegion      = 72,
    VMMDevReq_GetSeamlessChangeRequest   = 73,
    VMMDevReq_QueryCredentials           = 100,
    VMMDevReq_ReportCredentialsJudgement = 101,
    VMMDevReq_ReportGuestStats           = 110,
    VMMDevReq_GetMemBalloonChangeRequest = 111,
    VMMDevReq_GetStatisticsChangeRequest = 112,
    VMMDevReq_ChangeMemBalloon           = 113,
    VMMDevReq_GetVRDPChangeRequest       = 150,
    VMMDevReq_LogString                  = 200,
    VMMDevReq_GetCpuHotPlugRequest       = 210,
    VMMDevReq_SetCpuHotPlugStatus        = 211,
    VMMDevReq_RegisterSharedModule       = 212,
    VMMDevReq_UnregisterSharedModule     = 213,
    VMMDevReq_CheckSharedModules         = 214,
    VMMDevReq_GetPageSharingStatus       = 215,
    VMMDevReq_DebugIsPageShared          = 216,
    VMMDevReq_GetSessionId               = 217, /**< @since version 3.2.8 */
    VMMDevReq_WriteCoreDump              = 218,
    VMMDevReq_GuestHeartbeat             = 219,
    VMMDevReq_HeartbeatConfigure         = 220,
    VMMDevReq_NtBugCheck                 = 221,
    VMMDevReq_VideoUpdateMonitorPositions= 222,
    VMMDevReq_GetMouseStatusEx           = 223,
    VMMDevReq_SizeHack                   = 0x7fffffff
} VMMDevRequestType;

/** Version of VMMDevRequestHeader structure. */
#define VMMDEV_REQUEST_HEADER_VERSION (0x10001)

/**
 * Generic VMMDev request header.
 *
 * This structure is copied/mirrored by VBGLREQHDR in the VBoxGuest I/O control
 * interface.  Changes there needs to be mirrored in it.
 *
 * @sa VBGLREQHDR
 */
typedef struct VMMDevRequestHeader
{
    /** IN: Size of the structure in bytes (including body).
     * (VBGLREQHDR uses this for input size and output if reserved1 is zero). */
    uint32_t size;
    /** IN: Version of the structure.  */
    uint32_t version;
    /** IN: Type of the request.
     * @note VBGLREQHDR uses this for optional output size. */
    VMMDevRequestType requestType;
    /** OUT: VBox status code. */
    int32_t  rc;
    /** Reserved field no.1. MBZ.
     * @note VBGLREQHDR uses this for optional output size, however never for a
     *       real VMMDev request, only in the I/O control interface. */
    uint32_t reserved1;
    /** IN: Requestor information (VMMDEV_REQUESTOR_XXX) when
     * VBOXGSTINFO2_F_REQUESTOR_INFO is set, otherwise ignored by the host. */
    uint32_t fRequestor;
} VMMDevRequestHeader;
//AssertCompileSize(VMMDevRequestHeader, 24);

/**
 * HGCM request header.
 */
typedef struct VMMDevHGCMRequestHeader
{
    /** Request header. */
    VMMDevRequestHeader header;

    /** HGCM flags. */
    uint32_t fu32Flags;

    /** Result code. */
    int32_t result;
} VMMDevHGCMRequestHeader;
//AssertCompileSize(VMMDevHGCMRequestHeader, 24+8);


/**
 * HGCM connect request structure.
 *
 * Used by VMMDevReq_HGCMConnect.
 */
typedef struct
{
    /** HGCM request header. */
    VMMDevHGCMRequestHeader header;

    /** IN: Description of service to connect to. */
    HGCMServiceLocation loc;

    /** OUT: Client identifier assigned by local instance of HGCM. */
    uint32_t u32ClientID;
} VMMDevHGCMConnect;
//AssertCompileSize(VMMDevHGCMConnect, 32+132+4);

/**
 * HGCM call request structure.
 *
 * Used by VMMDevReq_HGCMCall32 and VMMDevReq_HGCMCall64.
 */
typedef struct
{
    /* request header */
    VMMDevHGCMRequestHeader header;

    /** IN: Client identifier. */
    uint32_t u32ClientID;
    /** IN: Service function number. */
    uint32_t u32Function;
    /** IN: Number of parameters. */
    uint32_t cParms;
    /** Parameters follow in form: HGCMFunctionParameter aParms[X]; */
} VMMDevHGCMCall;
//AssertCompileSize(VMMDevHGCMCall, 32+12);


/**
 * HGCM parameter type.
 */
typedef enum
{
    VMMDevHGCMParmType_Invalid            = 0,
    VMMDevHGCMParmType_32bit              = 1,
    VMMDevHGCMParmType_64bit              = 2,
    VMMDevHGCMParmType_PhysAddr           = 3,  /**< @deprecated Doesn't work, use PageList. */
    VMMDevHGCMParmType_LinAddr            = 4,  /**< In and Out */
    VMMDevHGCMParmType_LinAddr_In         = 5,  /**< In  (read;  host<-guest) */
    VMMDevHGCMParmType_LinAddr_Out        = 6,  /**< Out (write; host->guest) */
    VMMDevHGCMParmType_LinAddr_Locked     = 7,  /**< Locked In and Out - for VBoxGuest, not host. */
    VMMDevHGCMParmType_LinAddr_Locked_In  = 8,  /**< Locked In  (read;  host<-guest) - for VBoxGuest, not host. */
    VMMDevHGCMParmType_LinAddr_Locked_Out = 9,  /**< Locked Out (write; host->guest) - for VBoxGuest, not host. */
    VMMDevHGCMParmType_PageList           = 10, /**< Physical addresses of locked pages for a buffer. */
    VMMDevHGCMParmType_Embedded           = 11, /**< Small buffer embedded in request. */
    VMMDevHGCMParmType_ContiguousPageList = 12, /**< Like PageList but with physically contiguous memory, so only one page entry. */
    VMMDevHGCMParmType_NoBouncePageList   = 13, /**< Like PageList but host function requires no bounce buffering. */
    VMMDevHGCMParmType_SizeHack           = 0x7fffffff
} HGCMFunctionParameterType;
//AssertCompileSize(HGCMFunctionParameterType, 4);

/**
 * HGCM function parameter, 32-bit client.
 */
#  pragma pack(4) /* We force structure dword packing here for hysterical raisins.  Saves us 4 bytes, at the cost of
                     misaligning the value64 member of every other parameter structure. */
typedef struct HGCMFunctionParameter32
{
    HGCMFunctionParameterType type;
    union
    {
        uint32_t   value32;
        uint64_t   value64;
        struct
        {
            uint32_t size;

            union
            {
                RTGCPHYS32 physAddr;
                RTGCPTR32  linearAddr;
            } u;
        } Pointer;
        struct
        {
            uint32_t  cb;
            RTGCPTR32 uAddr;
        } LinAddr;                      /**< Shorter version of the above Pointer structure. */
        struct
        {
            uint32_t size;              /**< Size of the buffer described by the page list. */
            uint32_t offset;            /**< Relative to the request header of a HGCMPageListInfo structure, valid if size != 0. */
        } PageList;
        struct
        {
            uint32_t fFlags : 8;        /**< VBOX_HGCM_F_PARM_*. */
            uint32_t offData : 24;      /**< Relative to the request header, valid if cb != 0. */
            uint32_t cbData;            /**< The buffer size. */
        } Embedded;
    } u;
} HGCMFunctionParameter32;
#pragma pack()
//AssertCompileSize(HGCMFunctionParameter32, 4+8);

/**
 * HGCM function parameter, 64-bit client.
 */
#  pragma pack(4)/* We force structure dword packing here for hysterical raisins.  Saves us 4 bytes,
                    at the cost of misaligning the value64 members. */
typedef struct HGCMFunctionParameter64
{
    HGCMFunctionParameterType type;
    union
    {
        uint32_t   value32;
        uint64_t   value64;
        struct
        {
            uint32_t size;

            union
            {
                RTGCPHYS64 physAddr;
                RTGCPTR64  linearAddr;
            } u;
        } Pointer;
        struct
        {
            uint32_t  cb;
            RTGCPTR64 uAddr;
        } LinAddr;                      /**< Shorter version of the above Pointer structure. */
        struct
        {
            uint32_t size;              /**< Size of the buffer described by the page list. */
            uint32_t offset;            /**< Relative to the request header, valid if size != 0. */
        } PageList;
        struct
        {
            uint32_t fFlags : 8;        /**< VBOX_HGCM_F_PARM_*. */
            uint32_t offData : 24;      /**< Relative to the request header, valid if cb != 0. */
            uint32_t cbData;            /**< The buffer size. */
        } Embedded;
    } u;
} HGCMFunctionParameter64;
#pragma pack()
//AssertCompileSize(HGCMFunctionParameter64, 4+12);

/** @name The service functions which are called by guest.
 *
 * @note The numbers may not change!
 * @{
 */
/** Get a guest property */
#define GUEST_PROP_FN_GET_PROP              1
/** Set a guest property */
#define GUEST_PROP_FN_SET_PROP              2
/** Set just the value of a guest property */
#define GUEST_PROP_FN_SET_PROP_VALUE        3
/** Delete a guest property */
#define GUEST_PROP_FN_DEL_PROP              4
/** Enumerate guest properties */
#define GUEST_PROP_FN_ENUM_PROPS            5
/** Poll for guest notifications */
#define GUEST_PROP_FN_GET_NOTIFICATION      6
/** @} */

#endif
