#!/usr/bin/python

# ***************************************************************************
# *   Copyright (C) 2009 by Jesus Arias Fisteus                             *
# *   jaf@it.uc3m.es                                                        *
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU General Public License as published by  *
# *   the Free Software Foundation; either version 2 of the License, or     *
# *   (at your option) any later version.                                   *
# *                                                                         *
# *   This program is distributed in the hope that it will be useful,       *
# *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
# *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
# *   GNU General Public License for more details.                          *
# *                                                                         *
# *   You should have received a copy of the GNU General Public License     *
# *   along with this program; if not, write to the                         *
# *   Free Software Foundation, Inc.,                                       *
# *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
# ***************************************************************************

import sys

class Alias:
    def __init__(self, alias, charset):
        self.alias = alias
        self.charset = charset
        self.position = 0

    def __cmp__(self, other):
        return cmp(self.alias, other.alias)

class Charset:
    def __init__(self, name):
        self.name = name
        self.preferred_mime = ""
        self.iconv_name = ""
        self.aliases = []

    def find_iconv_name(self, csets_iconv):
        if self.preferred_mime in csets_iconv:
            self.iconv_name = self.preferred_mime
        elif self.name in csets_iconv:
            self.iconv_name = self.name
        else:
            for alias in self.aliases:
                if alias in csets_iconv:
                    self.iconv_name = alias
                    break

    def preferred_name(self):
        if self.preferred_mime != "":
            return self.preferred_mime
        else:
            return self.name

    def is_iconv_supported(self):
        return self.iconv_name != ""

def dump_output(aliases, output_dir):

    out_c = open(output_dir + "/charset_aliases.c", "w");
    out_h = open(output_dir + "/charset_aliases.h", "w");
    
    out_c.write("""/*
 * This file has been automatically generated by charsets_gen.py.
 * Do not edit it by hand.
 *
 * Definitions of aliases for all the charsets supported by the
 * application, obtained by mixing the IANA charset list [1] and
 * the list of iconv-supported charsets.
 *
 * [1] http://www.iana.org/assignments/character-sets
 *
 */

#include \"charset.h\"

""")

    buffer = ""
    pos = 0
    lpos = 0
    for alias in aliases:
        alias.position = pos
        alias_len = len(alias.alias) + 1
        pos = pos + alias_len
        lpos = lpos + alias_len
        if lpos > 70:
            buffer = buffer + "\\\n"
            lpos = alias_len
        buffer = buffer + alias.alias + "\\0"

    out_c.write("char charset_buffer[] = \"\\\n")
    out_c.write(buffer + "\";\n\n")

    out_c.write("charset_t charset_aliases[] = {\n")
    line = ""
    for alias in aliases:
        if line != "":
            out_c.write("    " + line + ",\n")
        preferred = locate_alias(alias.charset.preferred_name(), aliases)
        iconv = locate_alias(alias.charset.iconv_name, aliases)
        names = (alias.alias, preferred.alias, iconv.alias)
        out_c.write("    /* %s; %s; %s */\n"%names)
        positions = (alias.position, preferred.position, iconv.position)
        line = "{&charset_buffer[%d], &charset_buffer[%d], &charset_buffer[%d]}"\
               %positions
    out_c.write("    " + line + "\n};\n")
    out_c.close()

    out_h.write("""/*
 * This file has been automatically generated by charsets_gen.py.
 * Do not edit it by hand.
 *
 * Definitions of aliases for all the charsets supported by the
 * application, obtained by mixing the IANA charset list [1] and
 * the list of iconv-supported charsets.
 *
 * [1] http://www.iana.org/assignments/character-sets
 *
 */

""")
    out_h.write("/* Number of known aliases */\n")
    out_h.write("#define CHARSET_ALIAS_NUM %d\n\n"%len(aliases))
    out_h.write("/* Positions of aliases for some relevant character sets*/\n")

    latin1_pos = alias_position("iso-8859-1", aliases)
    utf8_pos = alias_position("utf-8", aliases)
    utf16_pos = alias_position("utf-16", aliases)
    utf16be_pos = alias_position("utf-16be", aliases)
    utf16le_pos = alias_position("utf-16le", aliases)
    ucs4_pos = alias_position("iso-10646-ucs-4", aliases)
    out_h.write("#define CHARSET_ISO_8859_1 &charset_aliases[%d]\n"%latin1_pos)
    out_h.write("#define CHARSET_UTF_8 &charset_aliases[%d]\n"%utf8_pos)
    out_h.write("#define CHARSET_UTF_16 &charset_aliases[%d]\n"%utf16_pos)
    out_h.write("#define CHARSET_UTF_16BE &charset_aliases[%d]\n"%utf16be_pos)
    out_h.write("#define CHARSET_UTF_16LE &charset_aliases[%d]\n"%utf16le_pos)
    out_h.write("#define CHARSET_UCS_4 &charset_aliases[%d]\n"%ucs4_pos)
    out_h.close()

def locate_alias(alias, aliases):
    for a in aliases:
        if a.alias == alias:
            return a
    return None

def alias_position(alias, aliases):
    i = 0
    for a in aliases:
        if a.alias == alias:
            return i
        i = i + 1
    return -1

def main():
    # some hardcoded aliases
    hardcoded_aliases = dict()
    hardcoded_aliases["utf-8"] = ["utf8"]
    hardcoded_aliases["utf-16"] = ["utf16"]
    hardcoded_aliases["utf-7"] = ["utf7"]
    hardcoded_aliases["iso-8859-1"] = ["iso-latin-1"]
    
    # Process the list of character set labels obtained from "iconv -l"
    csets_iconv = []
    iconv_file = open(sys.argv[2])
    for line in iconv_file:
        # strip the final "//"
        if len(line) > 0:
            alias = line[0:-3].lower()
            csets_iconv.append(alias)

    # Process list of IANA character sets
    csets = []
    iana_file = open(sys.argv[1])
    for line in iana_file:
        process_line = False
        if line.startswith("Name:"):
            # new character set
            alias = line.split()[1].lower()
            charset = Charset(alias)
            csets.append(charset)
            process_line = True
        elif line.startswith("Alias:"):
            alias = line.split()[1].lower()
            if alias != "none":
                process_line = True
        if process_line:
            if line.find("(preferred MIME name)") != -1:
                charset.preferred_mime = alias
            elif alias != charset.name:
                charset.aliases.append(alias)

    for charset in csets:
        charset.find_iconv_name(csets_iconv)
        if charset.name in hardcoded_aliases:
            charset.aliases.extend(hardcoded_aliases[charset.name])
        elif charset.preferred_mime in hardcoded_aliases:
            charset.aliases.extend(hardcoded_aliases[charset.preferred_mime])

    # Prepare the list of charsets to be handled
    aliases = []
    for charset in csets:
        if charset.is_iconv_supported():
            aliases.append(Alias(charset.name, charset))
            if charset.preferred_mime != "" \
               and charset.preferred_mime != charset.name:
                aliases.append(Alias(charset.preferred_mime, charset))
            for alias in charset.aliases:
                aliases.append(Alias(alias, charset))
    aliases.sort()

    dump_output(aliases, sys.argv[3])

#    for charset in csets:
#        if len(charset.iconv_name) == 0:
#            print "Unsupported: " + charset.name

#    for charset in csets:
#        print "Charset:", charset.name
#        print "Preferred:", charset.preferred_mime
#        print "Iconv:", charset.iconv_name
#        print "Aliases:", charset.aliases

def usage():
    print sys.argv[0], "iana_character_sets_file iconv_character_sets_file output_dir"

if __name__ == "__main__": # we're running as a script, not imported...
    main()

