// Comprehensive test for CVE-2022-24434 fix
const fetch = require('node-fetch');

async function testRequest(name, body, expectError = false) {
  try {
    console.log(`\n=== Testing: ${name} ===`);
    
    const response = await fetch('http://127.0.0.1:3000', {
      method: 'POST',
      headers: {
        'content-type': 'multipart/form-data; boundary=----WebKitFormBoundaryoo6vortfDzBsDiro',
        'connection': 'keep-alive'
      },
      body: body
    });

    console.log('Response status:', response.status);
    const text = await response.text();
    console.log('Response body:', text);
    console.log('✅ Request completed without server crash');
    
  } catch (error) {
    console.error('❌ Request failed:', error.message);
  }
}

async function runTests() {
  console.log('🧪 Running comprehensive CVE-2022-24434 tests...\n');
  
  // Test 1: Original CVE payload (unclosed part)
  await testRequest(
    'Original CVE - Unclosed part',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="test"\r\n\r\nUnclosed part'
  );
  
  // Test 2: Your specific POC
  await testRequest(
    'Your POC - Folded header',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\n Content-Disposition: form-data; name="bildbeschreibung"\r\n\r\n\r\n------WebKitFormBoundaryoo6vortfDzBsDiro--'
  );
  
  // Test 3: Valid multipart data
  await testRequest(
    'Valid multipart data',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="test"\r\n\r\nValid data\r\n------WebKitFormBoundaryoo6vortfDzBsDiro--\r\n'
  );
  
  // Test 4: Malformed boundary
  await testRequest(
    'Malformed boundary',
    '----WRONGBOUNDARY\r\nContent-Disposition: form-data; name="test"\r\n\r\nSome data'
  );
  
  // Test 5: Empty body
  await testRequest(
    'Empty body',
    ''
  );
  
  // Test 6: Multiple unclosed parts
  await testRequest(
    'Multiple unclosed parts',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="part1"\r\n\r\nData1\r\n------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="part2"\r\n\r\nData2'
  );
  
  // Test 7: Boundary without proper line endings
  await testRequest(
    'Malformed line endings',
    '------WebKitFormBoundaryoo6vortfDzBsDiroContent-Disposition: form-data; name="test"Data without proper separators'
  );
  
  // Test 8: Premature boundary termination
  await testRequest(
    'Premature boundary termination',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="test"\r\n\r\nSome data\r\n------WebKitFormBoundaryoo6vortfDzBsDiro'
  );
  
  // Test 9: Nested boundary attack
  await testRequest(
    'Nested boundary confusion',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="test"\r\n\r\n------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="nested"'
  );
  
  // Test 10: Extra long header without proper termination
  await testRequest(
    'Oversized header attack',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="' + 'A'.repeat(10000) + '"\r\n\r\ndata'
  );
  
  // Test 11: Boundary with only dashes (potential infinite loop)
  await testRequest(
    'Boundary with extra dashes',
    '------WebKitFormBoundaryoo6vortfDzBsDiro----\r\nContent-Disposition: form-data; name="test"\r\n\r\ndata'
  );
  
  // Test 12: Null bytes injection
  await testRequest(
    'Null bytes injection',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="test"\r\n\r\ndata\x00\x00\x00'
  );
  
  // Test 13: Unicode/UTF-8 boundary confusion
  await testRequest(
    'Unicode boundary confusion',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="tëst"\r\n\r\n🚀💥data'
  );
  
  // Test 14: Missing final CRLF before boundary
  await testRequest(
    'Missing CRLF before boundary',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="test"\r\n\r\ndata------WebKitFormBoundaryoo6vortfDzBsDiro--'
  );
  
  // Test 15: Binary data without proper encoding
  await testRequest(
    'Binary data injection',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="binary"\r\nContent-Type: application/octet-stream\r\n\r\n' + 
    Buffer.from([0xFF, 0xFE, 0xFD, 0xFC, 0x00, 0x01, 0x02, 0x03]).toString('binary')
  );
  
  // Test 16: Rapid fire unclosed parts (DoS simulation)
  console.log('\n🔥 Running DoS simulation test...');
  for (let i = 0; i < 5; i++) {
    await testRequest(
      `DoS simulation ${i + 1}/5`,
      '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="dos' + i + '"\r\n\r\nUnclosed part ' + i
    );
  }
  
  // Test 17: Memory exhaustion attempt with large data
  await testRequest(
    'Large payload test',
    '------WebKitFormBoundaryoo6vortfDzBsDiro\r\nContent-Disposition: form-data; name="large"\r\n\r\n' + 'X'.repeat(100000)
  );
  
  console.log('\n🎉 All advanced edge case tests completed! Server remained stable.');
}

runTests();
