#!/bin/bash


set -e

CONTAINER_NAME="zipslip-vulnerable-server"
IMAGE_NAME="zipslip-demo"
PORT="8080"

echo "Zip Slip Vulnerability Demo"
echo "================================================"
echo ""

# Check if Docker is installed
if ! command -v docker &> /dev/null; then
    echo "Docker is not installed. Please install Docker first."
    exit 1
fi

# Check if Docker is running
if ! docker info &> /dev/null; then
    echo "Docker is not running. Please start Docker first."
    exit 1
fi

echo "Docker is available"

# Create directories if they don't exist
mkdir -p uploads extracted

# Function to cleanup existing container and image
cleanup() {
    echo "Cleaning up existing container and image..."
    
    # Stop and remove container if it exists
    if docker ps -a --format "table {{.Names}}" | grep -q "^${CONTAINER_NAME}$"; then
        echo "Stopping existing container..."
        docker stop ${CONTAINER_NAME} 2>/dev/null || true
        echo "Removing existing container..."
        docker rm ${CONTAINER_NAME} 2>/dev/null || true
    fi
    
    # Remove image if it exists
    if docker images --format "table {{.Repository}}" | grep -q "^${IMAGE_NAME}$"; then
        echo "Removing existing image..."
        docker rmi ${IMAGE_NAME} 2>/dev/null || true
    fi
    
    echo "Cleanup completed"
    echo ""
}

# Function to build and run container
run_container() {
    echo "Building and running with Docker..."
    echo ""
    
    # Build the image
    echo "Building Docker image..."
    docker build -t ${IMAGE_NAME} .
    
    echo ""
    echo "Starting container..."
    docker run -d \
        --name ${CONTAINER_NAME} \
        -p ${PORT}:8080 \
        ${IMAGE_NAME}
    
    echo ""
    echo "Container is running!"
    echo "Open http://localhost:${PORT} in your browser"
    echo ""
    echo "Uploaded files will be saved to: ./uploads"
    echo "Extracted files will be saved to: ./extracted"
    echo ""
    echo "Container Management:"
    echo "  Stop container:    docker stop ${CONTAINER_NAME}"
    echo "  View logs:         docker logs -f ${CONTAINER_NAME}"
    echo "  Enter container:   docker exec -it ${CONTAINER_NAME} /bin/bash"
    echo "  Remove container:  docker rm ${CONTAINER_NAME}"
}

# Function to show usage
show_usage() {
    echo "Usage: $0 [start|stop|cleanup|logs|status]"
    echo ""
    echo "Commands:"
    echo "  start     - Build and start the container (default)"
    echo "  stop      - Stop the running container"
    echo "  cleanup   - Remove container and image"
    echo "  logs      - Show container logs"
    echo "  status    - Show container status"
    echo "  help      - Show this help message"
    echo ""
    echo "Examples:"
    echo "  $0 start"
    echo "  $0 stop"
    echo "  $0 cleanup"
    echo "  $0 logs"
}

# Function to stop container
stop_container() {
    echo "Stopping container..."
    if docker ps --format "table {{.Names}}" | grep -q "^${CONTAINER_NAME}$"; then
        docker stop ${CONTAINER_NAME}
        echo "Container stopped"
    else
        echo "Container is not running"
    fi
}

# Function to show logs
show_logs() {
    echo "Container logs:"
    if docker ps -a --format "table {{.Names}}" | grep -q "^${CONTAINER_NAME}$"; then
        docker logs -f ${CONTAINER_NAME}
    else
        echo "Container not found"
    fi
}

# Function to show status
show_status() {
    echo "Container status:"
    if docker ps --format "table {{.Names}}" | grep -q "^${CONTAINER_NAME}$"; then
        echo "Container is running"
        docker ps --filter "name=${CONTAINER_NAME}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
    elif docker ps -a --format "table {{.Names}}" | grep -q "^${CONTAINER_NAME}$"; then
        echo "Container exists but is not running"
        docker ps -a --filter "name=${CONTAINER_NAME}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
    else
        echo "Container not found"
    fi
}

# Main script logic
case "${1:-start}" in
    "start")
        cleanup
        run_container
        ;;
    "stop")
        stop_container
        ;;
    "cleanup")
        cleanup
        ;;
    "logs")
        show_logs
        ;;
    "status")
        show_status
        ;;
    "help"|"-h"|"--help")
        show_usage
        ;;
    *)
        echo "Unknown command: $1"
        show_usage
        exit 1
        ;;
esac
