#!/usr/bin/env python3

import sys
import os
import unittest
import time
import tempfile
from unittest.mock import Mock, patch, MagicMock

try:
    from core import *
    from payloads import *
    from evasion import *
    from monitor import *
    from multiarch import *
    from framework import *
except ImportError as e:
    print(f"[-] Module import failed: {e}")
    print("    Ensure all files are in the same directory")
    sys.exit(1)

class TestSystemDetector(unittest.TestCase):
    def setUp(self):
        self.detector = SystemDetector()
        
    def test_get_windows_version(self):
        version = self.detector.get_windows_version()
        self.assertIsInstance(version, str)
        self.assertNotEqual(version, "")
        
    def test_is_64bit(self):
        result = self.detector.is_64bit()
        self.assertIsInstance(result, bool)
        
    def test_check_driver_present(self):
        result = self.detector.check_driver_present()
        self.assertIsInstance(result, bool)
        
    def test_check_vbs_enabled(self):
        result = self.detector.check_vbs_enabled()
        self.assertIsInstance(result, bool)
        
    def test_check_core_isolation(self):
        result = self.detector.check_core_isolation()
        self.assertIsInstance(result, bool)

class TestDefenseEvasion(unittest.TestCase):
    def setUp(self):
        self.evasion = DefenseEvasion()
        
    def test_anti_debug_check(self):
        result = self.evasion.anti_debug_check()
        self.assertIsInstance(result, bool)
        
    def test_obfuscate_string(self):
        test_data = b"test_string"
        obfuscated, key = self.evasion.obfuscate_string(test_data)
        
        self.assertIsInstance(obfuscated, bytes)
        self.assertIsInstance(key, int)
        self.assertNotEqual(obfuscated, test_data)
        
        deobfuscated = self.evasion.deobfuscate_string(obfuscated, key)
        self.assertEqual(deobfuscated, test_data)

class TestKernelExploiter(unittest.TestCase):
    def setUp(self):
        self.exploiter = KernelExploiter()
        
    @patch('ctypes.windll.kernel32.CreateFileW')
    def test_connect_driver_success(self, mock_create_file):
        mock_create_file.return_value = 123
        result = self.exploiter.connect_driver()
        self.assertTrue(result)
        
    @patch('ctypes.windll.kernel32.CreateFileW')
    def test_connect_driver_failure(self, mock_create_file):
        mock_create_file.return_value = -1
        result = self.exploiter.connect_driver()
        self.assertFalse(result)
        
    def test_read_msr_no_handle(self):
        result = self.exploiter.read_msr(0xC0000082)
        self.assertIsNone(result)
        
    def test_write_msr_no_handle(self):
        result = self.exploiter.write_msr(0xC0000082, 0x12345678)
        self.assertFalse(result)

class TestShellcodeGenerator(unittest.TestCase):
    def setUp(self):
        self.generator = ShellcodeGenerator()
        
    def test_generate_token_steal_x64(self):
        shellcode = self.generator.generate_token_steal_x64()
        self.assertIsInstance(shellcode, bytes)
        self.assertGreater(len(shellcode), 0)
        
    def test_generate_token_steal_x86(self):
        shellcode = self.generator.generate_token_steal_x86()
        self.assertIsInstance(shellcode, bytes)
        self.assertGreater(len(shellcode), 0)
        
    def test_generate_token_steal_arm64(self):
        shellcode = self.generator.generate_token_steal_arm64()
        self.assertIsInstance(shellcode, bytes)
        self.assertGreater(len(shellcode), 0)
        
    def test_generate_privilege_escalation_x64(self):
        shellcode = self.generator.generate_privilege_escalation_x64()
        self.assertIsInstance(shellcode, bytes)
        self.assertGreater(len(shellcode), 0)
        
    def test_generate_disable_smep_x64(self):
        shellcode = self.generator.generate_disable_smep_x64()
        self.assertIsInstance(shellcode, bytes)
        self.assertGreater(len(shellcode), 0)

class TestCustomPayloadBuilder(unittest.TestCase):
    def setUp(self):
        self.builder = CustomPayloadBuilder()
        
    def test_build_token_steal_payload_x64(self):
        payload = self.builder.build_custom_payload("token_steal", "x64")
        self.assertIsInstance(payload, bytes)
        self.assertGreater(len(payload), 0)
        
    def test_build_token_steal_payload_x86(self):
        payload = self.builder.build_custom_payload("token_steal", "x86")
        self.assertIsInstance(payload, bytes)
        self.assertGreater(len(payload), 0)
        
    def test_build_privilege_escalation_payload(self):
        payload = self.builder.build_custom_payload("privilege_escalation", "x64")
        self.assertIsInstance(payload, bytes)
        self.assertGreater(len(payload), 0)
        
    def test_build_reverse_shell_payload(self):
        options = {"host": "127.0.0.1", "port": 4444}
        payload = self.builder.build_custom_payload("reverse_shell", "x64", options)
        self.assertIsInstance(payload, bytes)
        self.assertGreater(len(payload), 0)
        
    def test_build_invalid_payload_type(self):
        payload = self.builder.build_custom_payload("invalid_type", "x64")
        self.assertIsNone(payload)

class TestPayloadEncoder(unittest.TestCase):
    def setUp(self):
        self.encoder = PayloadEncoder()
        
    def test_xor_encode(self):
        test_payload = b"test_payload_data"
        encoded, key = self.encoder.xor_encode(test_payload)
        
        self.assertIsInstance(encoded, bytes)
        self.assertIsInstance(key, int)
        self.assertNotEqual(encoded, test_payload)
        
        decoded, _ = self.encoder.xor_encode(encoded, key)
        self.assertEqual(decoded, test_payload)
        
    def test_shikata_ga_nai_encode(self):
        test_payload = b"test_payload_data"
        encoded, key = self.encoder.shikata_ga_nai_encode(test_payload)
        
        self.assertIsInstance(encoded, bytes)
        self.assertIsInstance(key, int)
        self.assertNotEqual(encoded, test_payload)
        
    def test_alpha_mixed_encode(self):
        test_payload = b"test_payload_data"
        encoded = self.encoder.alpha_mixed_encode(test_payload)
        
        self.assertIsInstance(encoded, bytes)
        
    def test_polymorphic_encode(self):
        test_payload = b"test_payload_data"
        encoded = self.encoder.polymorphic_encode(test_payload)
        
        self.assertIsInstance(encoded, bytes)
        self.assertGreaterEqual(len(encoded), len(test_payload))

class TestExploitPayloads(unittest.TestCase):
    def setUp(self):
        self.payloads = ExploitPayloads()
        
    def test_get_token_steal_payload(self):
        payload = self.payloads.get_token_steal_payload("x64")
        self.assertIsInstance(payload, bytes)
        self.assertGreater(len(payload), 0)
        
    def test_get_privilege_escalation_payload(self):
        payload = self.payloads.get_privilege_escalation_payload("x64")
        self.assertIsInstance(payload, bytes)
        self.assertGreater(len(payload), 0)
        
    def test_get_reverse_shell_payload(self):
        payload = self.payloads.get_reverse_shell_payload("127.0.0.1", 4444, "x64")
        self.assertIsInstance(payload, bytes)
        self.assertGreater(len(payload), 0)
        
    def test_get_meterpreter_payload(self):
        payload = self.payloads.get_meterpreter_payload("127.0.0.1", 4444, "x64")
        self.assertIsInstance(payload, bytes)
        self.assertGreater(len(payload), 0)
        
    def test_get_encoded_payload(self):
        payload = self.payloads.get_token_steal_payload("x64", encoded=True)
        self.assertIsInstance(payload, bytes)
        self.assertGreater(len(payload), 0)

class TestAdvancedEvasion(unittest.TestCase):
    def setUp(self):
        self.evasion = AdvancedEvasion()
        
    def test_detect_vm(self):
        result = self.evasion.detect_vm()
        self.assertIsInstance(result, bool)
        
    def test_detect_analysis_tools(self):
        result = self.evasion.detect_analysis_tools()
        self.assertIsInstance(result, bool)
        
    def test_check_mouse_activity(self):
        result = self.evasion.check_mouse_activity()
        self.assertIsInstance(result, bool)
        
    def test_check_uptime(self):
        result = self.evasion.check_uptime()
        self.assertIsInstance(result, bool)
        
    def test_check_memory_size(self):
        result = self.evasion.check_memory_size()
        self.assertIsInstance(result, bool)
        
    def test_check_cpu_cores(self):
        result = self.evasion.check_cpu_cores()
        self.assertIsInstance(result, bool)
        
    def test_timing_evasion(self):
        result = self.evasion.timing_evasion()
        self.assertIsInstance(result, bool)
        
    def test_sleep_evasion(self):
        start_time = time.time()
        result = self.evasion.sleep_evasion()
        end_time = time.time()
        
        self.assertIsInstance(result, bool)
        self.assertGreaterEqual(end_time - start_time, 4.5)
        
    def test_comprehensive_env_check(self):
        result = self.evasion.comprehensive_env_check()
        self.assertIsInstance(result, bool)

class TestMultiArchSupport(unittest.TestCase):
    def setUp(self):
        self.multiarch = MultiArchSupport()
        
    def test_detect_architecture(self):
        arch = self.multiarch.detect_architecture()
        self.assertIn(arch, ["x86", "x64", "arm64", "unknown"])
        
    def test_get_pointer_size(self):
        size_x86 = self.multiarch.get_pointer_size("x86")
        size_x64 = self.multiarch.get_pointer_size("x64")
        size_arm64 = self.multiarch.get_pointer_size("arm64")
        
        self.assertEqual(size_x86, 4)
        self.assertEqual(size_x64, 8)
        self.assertEqual(size_arm64, 8)
        
    def test_get_register_context(self):
        context_x86 = self.multiarch.get_register_context("x86")
        context_x64 = self.multiarch.get_register_context("x64")
        context_arm64 = self.multiarch.get_register_context("arm64")
        
        self.assertIsInstance(context_x86, dict)
        self.assertIsInstance(context_x64, dict)
        self.assertIsInstance(context_arm64, dict)
        
        self.assertIn("instruction_pointer", context_x86)
        self.assertIn("instruction_pointer", context_x64)
        self.assertIn("instruction_pointer", context_arm64)
        
    def test_get_calling_convention(self):
        conv_x86 = self.multiarch.get_calling_convention("x86")
        conv_x64 = self.multiarch.get_calling_convention("x64")
        conv_arm64 = self.multiarch.get_calling_convention("arm64")
        
        self.assertIsInstance(conv_x86, dict)
        self.assertIsInstance(conv_x64, dict)
        self.assertIsInstance(conv_arm64, dict)
        
    def test_get_nop_instruction(self):
        nop_x86 = self.multiarch.get_nop_instruction("x86")
        nop_x64 = self.multiarch.get_nop_instruction("x64")
        nop_arm64 = self.multiarch.get_nop_instruction("arm64")
        
        self.assertIsInstance(nop_x86, bytes)
        self.assertIsInstance(nop_x64, bytes)
        self.assertIsInstance(nop_arm64, bytes)
        
    def test_pack_unpack_pointer(self):
        test_value = 0x12345678
        
        packed_x86 = self.multiarch.pack_pointer(test_value, "x86")
        unpacked_x86 = self.multiarch.unpack_pointer(packed_x86, "x86")
        
        self.assertEqual(unpacked_x86, test_value)
        
    def test_get_msr_support(self):
        support_x86 = self.multiarch.get_msr_support("x86")
        support_x64 = self.multiarch.get_msr_support("x64")
        support_arm64 = self.multiarch.get_msr_support("arm64")
        
        self.assertTrue(support_x86["supported"])
        self.assertTrue(support_x64["supported"])
        self.assertFalse(support_arm64["supported"])
        
    def test_is_architecture_supported(self):
        self.assertTrue(self.multiarch.is_architecture_supported("x86"))
        self.assertTrue(self.multiarch.is_architecture_supported("x64"))
        self.assertTrue(self.multiarch.is_architecture_supported("arm64"))
        self.assertFalse(self.multiarch.is_architecture_supported("invalid"))
        
    def test_validate_shellcode_for_arch(self):
        valid_shellcode = b"\x90\x90\x90\x90"
        invalid_shellcode = b""
        
        self.assertTrue(self.multiarch.validate_shellcode_for_arch(valid_shellcode, "x64"))
        self.assertFalse(self.multiarch.validate_shellcode_for_arch(invalid_shellcode, "x64"))

class TestFrameworkIntegration(unittest.TestCase):
    def setUp(self):
        self.framework = FrameworkIntegration()
        
    def test_generate_metasploit_payload(self):
        payload = self.framework.generate_metasploit_payload()
        self.assertIsInstance(payload, bytes)
        self.assertGreater(len(payload), 0)
        
    def test_generate_cobalt_strike_beacon(self):
        beacon = self.framework.generate_cobalt_strike_beacon()
        self.assertIsInstance(beacon, bytes)
        self.assertGreater(len(beacon), 0)
        
    def test_create_metasploit_module(self):
        module_path = self.framework.create_metasploit_module()
        self.assertIsInstance(module_path, str)
        self.assertTrue(os.path.exists(module_path))
        
        with open(module_path, 'r') as f:
            content = f.read()
            self.assertIn("CVE-2018-19323", content)
            self.assertIn("kali", content)
            self.assertIn("blueisbeautiful", content)
            
        os.unlink(module_path)
        
    def test_create_cobalt_strike_aggressor(self):
        script_path = self.framework.create_cobalt_strike_aggressor()
        self.assertIsInstance(script_path, str)
        self.assertTrue(os.path.exists(script_path))
        
        with open(script_path, 'r') as f:
            content = f.read()
            self.assertIn("CVE-2018-19323", content)
            self.assertIn("kali", content)
            self.assertIn("blueisbeautiful", content)
            
        os.unlink(script_path)
        
    def test_get_framework_status(self):
        status = self.framework.get_framework_status()
        self.assertIsInstance(status, dict)
        self.assertIn("metasploit", status)
        self.assertIn("cobalt_strike", status)

class TestExploitLogger(unittest.TestCase):
    def setUp(self):
        self.logger = ExploitLogger(debug=True)
        
    def test_logging_methods(self):
        self.logger.info("Test info message")
        self.logger.warning("Test warning message")
        self.logger.error("Test error message")
        self.logger.critical("Test critical message")
        self.logger.success("Test success message")
        self.logger.debug("Test debug message")
        
        logs = self.logger.get_logs()
        self.assertEqual(len(logs), 6)
        
        error_logs = self.logger.get_logs("ERROR")
        self.assertEqual(len(error_logs), 1)
        
    def test_save_logs(self):
        self.logger.info("Test message")
        log_file = self.logger.save_logs()
        
        self.assertIsNotNone(log_file)
        self.assertTrue(os.path.exists(log_file))
        
        with open(log_file, 'r') as f:
            data = json.load(f)
            self.assertIsInstance(data, list)
            self.assertGreater(len(data), 0)
            
        os.unlink(log_file)

class TestPerformanceMonitor(unittest.TestCase):
    def setUp(self):
        self.monitor = PerformanceMonitor()
        
    def test_timing_operations(self):
        operation = "test_operation"
        
        self.monitor.start_monitoring(operation)
        time.sleep(0.1)
        duration = self.monitor.end_monitoring(operation)
        
        self.assertIsNotNone(duration)
        self.assertGreaterEqual(duration, 0.1)
        
        avg_time = self.monitor.get_average_time(operation)
        total_time = self.monitor.get_total_time(operation)
        count = self.monitor.get_operation_count(operation)
        
        self.assertEqual(avg_time, duration)
        self.assertEqual(total_time, duration)
        self.assertEqual(count, 1)
        
    def test_get_all_stats(self):
        self.monitor.start_monitoring("op1")
        time.sleep(0.05)
        self.monitor.end_monitoring("op1")
        
        self.monitor.start_monitoring("op2")
        time.sleep(0.1)
        self.monitor.end_monitoring("op2")
        
        stats = self.monitor.get_all_stats()
        self.assertIsInstance(stats, dict)
        self.assertIn("op1", stats)
        self.assertIn("op2", stats)

class TestMetricsCollector(unittest.TestCase):
    def setUp(self):
        self.metrics = MetricsCollector()
        
    def test_increment_metrics(self):
        initial_value = self.metrics.get_metric("connection_attempts")
        self.metrics.increment("connection_attempts")
        new_value = self.metrics.get_metric("connection_attempts")
        
        self.assertEqual(new_value, initial_value + 1)
        
    def test_set_metric(self):
        self.metrics.set_metric("custom_metric", 42)
        value = self.metrics.get_metric("custom_metric")
        
        self.assertEqual(value, 42)
        
    def test_calculate_success_rates(self):
        self.metrics.increment("connection_attempts")
        self.metrics.increment("connection_attempts")
        self.metrics.increment("connection_successes")
        
        rates = self.metrics.calculate_success_rates()
        self.assertIn("connection_success_rate", rates)
        self.assertEqual(rates["connection_success_rate"], 50.0)

class TestExploitMonitor(unittest.TestCase):
    def setUp(self):
        self.monitor = ExploitMonitor(debug=True)
        
    def test_monitoring_lifecycle(self):
        self.monitor.start_monitoring()
        time.sleep(0.1)
        self.monitor.stop_monitoring()
        
        duration = self.monitor.get_session_duration()
        self.assertIsNotNone(duration)
        self.assertGreaterEqual(duration, 0.1)
        
    def test_generate_final_report(self):
        self.monitor.start_monitoring()
        self.monitor.metrics.increment("connection_attempts")
        self.monitor.metrics.increment("connection_successes")
        time.sleep(0.1)
        self.monitor.stop_monitoring()
        
        report = self.monitor.generate_final_report()
        
        self.assertIsInstance(report, dict)
        self.assertIn("session_info", report)
        self.assertIn("metrics", report)
        self.assertIn("performance", report)
        self.assertIn("logs", report)
        
        report_file = f"exploit_report_{self.monitor.logger.session_id}.json"
        self.assertTrue(os.path.exists(report_file))
        os.unlink(report_file)
        
        log_file = f"exploit_log_{self.monitor.logger.session_id}.json"
        if os.path.exists(log_file):
            os.unlink(log_file)

def run_unit_tests():
    print("Running CVE-2018-19323 Framework Unit Tests")
    print("=" * 50)
    
    test_classes = [
        TestSystemDetector,
        TestDefenseEvasion,
        TestKernelExploiter,
        TestShellcodeGenerator,
        TestCustomPayloadBuilder,
        TestPayloadEncoder,
        TestExploitPayloads,
        TestAdvancedEvasion,
        TestMultiArchSupport,
        TestFrameworkIntegration,
        TestExploitLogger,
        TestPerformanceMonitor,
        TestMetricsCollector,
        TestExploitMonitor
    ]
    
    total_tests = 0
    total_failures = 0
    total_errors = 0
    
    for test_class in test_classes:
        print(f"\nRunning {test_class.__name__}...")
        
        suite = unittest.TestLoader().loadTestsFromTestCase(test_class)
        runner = unittest.TextTestRunner(verbosity=0, stream=open(os.devnull, 'w'))
        result = runner.run(suite)
        
        total_tests += result.testsRun
        total_failures += len(result.failures)
        total_errors += len(result.errors)
        
        if result.failures or result.errors:
            print(f"  FAILED: {len(result.failures)} failures, {len(result.errors)} errors")
            for failure in result.failures:
                print(f"    FAIL: {failure[0]}")
            for error in result.errors:
                print(f"    ERROR: {error[0]}")
        else:
            print(f"  PASSED: {result.testsRun} tests")
    
    print("\n" + "=" * 50)
    print(f"Total Tests: {total_tests}")
    print(f"Passed: {total_tests - total_failures - total_errors}")
    print(f"Failed: {total_failures}")
    print(f"Errors: {total_errors}")
    
    if total_failures == 0 and total_errors == 0:
        print("ALL TESTS PASSED!")
        return True
    else:
        print("SOME TESTS FAILED!")
        return False

def run_integration_tests():
    print("\nRunning Integration Tests")
    print("=" * 30)
    
    try:
        print("[*] Testing module imports...")
        from core import SystemDetector, DefenseEvasion, KernelExploiter, PostExploit
        from payloads import ShellcodeGenerator, CustomPayloadBuilder, PayloadEncoder, ExploitPayloads
        from evasion import AdvancedEvasion, PersistenceManager, NetworkEvasion, PayloadDelivery
        from monitor import ExploitLogger, PerformanceMonitor, MetricsCollector, ExploitMonitor
        from multiarch import MultiArchSupport
        from framework import FrameworkIntegration
        print("[+] All modules imported successfully")
        
        print("[*] Testing basic functionality...")
        detector = SystemDetector()
        version = detector.get_windows_version()
        arch = detector.is_64bit()
        print(f"[+] System detection: Windows {version}, 64-bit: {arch}")
        
        print("[*] Testing payload generation...")
        payloads = ExploitPayloads()
        token_payload = payloads.get_token_steal_payload("x64")
        print(f"[+] Token steal payload generated: {len(token_payload)} bytes")
        
        print("[*] Testing multi-architecture support...")
        multiarch = MultiArchSupport()
        current_arch = multiarch.detect_architecture()
        print(f"[+] Current architecture: {current_arch}")
        
        print("[*] Testing monitoring system...")
        monitor = ExploitMonitor(debug=False)
        monitor.start_monitoring()
        monitor.metrics.increment("test_metric")
        monitor.stop_monitoring()
        print("[+] Monitoring system functional")
        
        print("[*] Testing framework integration...")
        framework = FrameworkIntegration()
        status = framework.get_framework_status()
        print(f"[+] Framework status: Metasploit={status['metasploit']['available']}, Cobalt Strike={status['cobalt_strike']['available']}")
        
        print("\n[+] All integration tests passed!")
        return True
        
    except Exception as e:
        print(f"\n[-] Integration test failed: {e}")
        return False

def run_performance_tests():
    print("\nRunning Performance Tests")
    print("=" * 30)
    
    try:
        print("[*] Testing payload generation performance...")
        payloads = ExploitPayloads()
        
        start_time = time.time()
        for i in range(100):
            payload = payloads.get_token_steal_payload("x64")
        end_time = time.time()
        
        avg_time = (end_time - start_time) / 100
        print(f"[+] Average payload generation time: {avg_time:.4f} seconds")
        
        print("[*] Testing evasion check performance...")
        evasion = AdvancedEvasion()
        
        start_time = time.time()
        result = evasion.comprehensive_env_check()
        end_time = time.time()
        
        check_time = end_time - start_time
        print(f"[+] Comprehensive evasion check time: {check_time:.4f} seconds")
        
        print("[*] Testing architecture detection performance...")
        multiarch = MultiArchSupport()
        
        start_time = time.time()
        for i in range(1000):
            arch = multiarch.detect_architecture()
        end_time = time.time()
        
        avg_time = (end_time - start_time) / 1000
        print(f"[+] Average architecture detection time: {avg_time:.6f} seconds")
        
        print("\n[+] All performance tests completed!")
        return True
        
    except Exception as e:
        print(f"\n[-] Performance test failed: {e}")
        return False

def main():
    print("CVE-2018-19323 GIGABYTE GDrv Advanced Exploitation Framework")
    print("Test Suite by kali (@blueisbeautiful)")
    print("=" * 70)
    
    if len(sys.argv) > 1:
        test_type = sys.argv[1].lower()
        
        if test_type == "--unit":
            success = run_unit_tests()
        elif test_type == "--integration":
            success = run_integration_tests()
        elif test_type == "--performance":
            success = run_performance_tests()
        elif test_type == "--all":
            unit_success = run_unit_tests()
            integration_success = run_integration_tests()
            performance_success = run_performance_tests()
            success = unit_success and integration_success and performance_success
        else:
            print(f"Unknown test type: {test_type}")
            print("Available options: --unit, --integration, --performance, --all")
            sys.exit(1)
    else:
        success = run_unit_tests()
    
    if success:
        print("\n🎉 ALL TESTS SUCCESSFUL! 🎉")
        sys.exit(0)
    else:
        print("\n❌ SOME TESTS FAILED! ❌")
        sys.exit(1)

if __name__ == "__main__":
    main()