// File: JMXFullDumper.java
import javax.management.*;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

public class JMXFullDumper {

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java JMXFullDumper <target_url> [filter_keyword]");
            System.out.println("Example: java JMXFullDumper service:jmx:rmi... Config");
            System.out.println("If no filter_keyword is provided, all MBeans will be dumped.");
            return;
        }

        String url = args[0];
        String filterKeyword = (args.length > 1) ? args[1] : ""; // Optional filter
        String username = "admin";
        String password = "admin";

        JMXConnector connector = null;
        try {
            JMXServiceURL serviceURL = new JMXServiceURL(url);
            Map<String, Object> env = new HashMap<>();
            String[] credentials = {username, password};
            env.put(JMXConnector.CREDENTIALS, credentials);

            System.out.println("[*] Connecting to JMX server at " + url);
            connector = JMXConnectorFactory.connect(serviceURL, env);
            MBeanServerConnection mbsc = connector.getMBeanServerConnection();
            System.out.println("[+] Successfully connected to JMX server.");

            // Get all MBean names and sort them for clean output
            Set<ObjectName> mbeanNames = new TreeSet<>(mbsc.queryNames(null, null));
            System.out.println("[*] Found " + mbeanNames.size() + " total MBeans.");
            System.out.println("\n--- Dumping MBean Details (filtering for '" + filterKeyword + "') ---");

            for (ObjectName name : mbeanNames) {
                // Apply the filter
                if (!filterKeyword.isEmpty() && !name.toString().toLowerCase().contains(filterKeyword.toLowerCase())) {
                    continue;
                }

                System.out.println("\n=======================================================================");
                System.out.println(" MBean: " + name);
                System.out.println("=======================================================================");

                try {
                    MBeanInfo mbeanInfo = mbsc.getMBeanInfo(name);

                    // --- 1. DUMP ATTRIBUTES ---
                    System.out.println("  [Attributes]");
                    MBeanAttributeInfo[] attributes = mbeanInfo.getAttributes();
                    if (attributes.length == 0) {
                        System.out.println("    - None");
                    }
                    for (MBeanAttributeInfo attr : attributes) {
                        String attrDetails = String.format("    - %s (%s)", attr.getName(), attr.getType());
                        if (attr.isReadable()) {
                            try {
                                Object value = mbsc.getAttribute(name, attr.getName());
                                // Handle array types for cleaner printing
                                String valueStr;
                                if (value instanceof String[]) {
                                    valueStr = Arrays.toString((String[]) value);
                                } else {
                                    valueStr = (value != null) ? value.toString() : "null";
                                }
                                System.out.println(attrDetails + " = " + valueStr);
                            } catch (Exception e) {
                                System.out.println(attrDetails + " = [Error reading value: " + e.getMessage() + "]");
                            }
                        } else {
                             System.out.println(attrDetails + " [Write-Only]");
                        }
                    }

                    // --- 2. DUMP OPERATIONS ---
                    System.out.println("\n  [Operations]");
                    MBeanOperationInfo[] operations = mbeanInfo.getOperations();
                    if (operations.length == 0) {
                        System.out.println("    - None");
                    }
                    for (MBeanOperationInfo op : operations) {
                        MBeanParameterInfo[] params = op.getSignature();
                        StringBuilder signature = new StringBuilder();
                        for (int i = 0; i < params.length; i++) {
                            signature.append(String.format("%s %s", params[i].getType(), params[i].getName()));
                            if (i < params.length - 1) {
                                signature.append(", ");
                            }
                        }
                        System.out.printf("    - %s %s(%s)\n", op.getReturnType(), op.getName(), signature.toString());
                    }

                } catch (Exception e) {
                    System.err.println("  [!] Could not retrieve full info for this MBean: " + e.getMessage());
                }
            }

        } catch (Exception e) {
            System.err.println("\n[!] An error occurred: " + e.getMessage());
            e.printStackTrace();
        } finally {
            if (connector != null) {
                try {
                    connector.close();
                    System.out.println("\n[*] Connection closed.");
                } catch (Exception e) {
                    System.err.println("Error closing connection: " + e.getMessage());
                }
            }
        }
    }
}
