#!/bin/sh

this_file="exec_cmd"		# don't change, used by ReportHandler()s

usage()
{
    echo "
$this_file.sh -c COMMAND -t TEXT_MSG [-i MORE_MSG] [-h]

  (COMMAND) Something like:

     ntfslabel -f /dev/sda1
     mount UUID=496BA79D0BC67E9C /mnt/496BA79D0BC67E9C -t ntfs -o noatime,
       nodiratime,force,big_writes
     insmod wl.ko

  (TEXT_MSG) 'TEXT_MSG_pass' and 'TEXT_MSG_fail' must be followed by ReportHnd()
             of the caller instance.

     ntfslabel_get
     disk_mount
     insmod_wl

  (MORE_MSG) may be used for any purposes. It is appended to the command result.

     UUID:496BA79D0BC67E9C (for example, this id field is used in ReportHnd()
                              to identify the partition which is owner of the
                              executed command)

  -h: Dump this help

  Example commands:

    $this_file.sh -c 'ntfslabel -f /dev/sda1' -t 'ntfslabel_get' -i 'UUID:496BA79D0BC67E9C'
    $this_file.sh -c 'mount UUID=496BA79D0BC67E9C /mnt/496BA79D0BC67E9C -t ntfs -o noatime,
      nodiratime,force,big_writes -t 'disk_mount' -i 'UUID:496BA79D0BC67E9C'
    $this_file.sh -c 'insmod /lib/modules/wireless/wl.ko name=wl0' -t 'insmod_wl' -i 'name:wl0'

  Example command results - echoes captured by ReportHnd()s:

     - Pay attention to the field key 'answer'.
     - UUID:496BA79D0BC67E9C is an additional message posted by caller, not used internally.
     - answer:xxxx includes the output of the executed command.
       * In the case of getting ntfs partition label, it is the label value
       * In the case of 'mounting', the value of this field is not necessary/meaningful; as a
         result, highly probably, it will not be used by the caller.

     exec_cmd: text_msg:ntfslabel_get_pass result:UUID:496BA79D0BC67E9C;answer:ntfs-seagate;
     exec_cmd: text_msg:disk_mount_pass result:UUID:496BA79D0BC67E9C;answer:xxx_not_important_xxx
"
}

#
# Inherit configuration file content
#

enable_stat=1
stat_max=1000
stat_file=/var/exec_cmd.stat

enable_log=1
log_max=10
log_file=/var/exec_cmd.log
log_start_key="Executing"

[ -f "/var/exec_cmd.conf" ] && {
    . /var/exec_cmd.conf
}

print_result()
{
    #
    # Don't change the following echo string, it is followed by ReportHnds()
    #
    echo "$this_file: text_msg:$1 result:$2"
}

log_init()
{
    echo "" > $log_file
    echo "*** LOG FILE for $this_file ***" >> $log_file
}

log_print()
{
    [ "$enable_log" != "1" ] && return

    [ ! -f $log_file ] && {
	log_init
    } || {
	is_start_key=`echo $1 | grep $log_start_key -c`
	[ $is_start_key = 1 ] && {
	    log_total=`grep $log_start_key -c $log_file`
	    [ $log_total -ge $log_max ] && {
		log_init
	    }
	}
    }

    [ "$is_first_run" != "no" ] && {
	echo >> $log_file
	is_first_run="no"
    }

    echo "$this_file: '${1}'" >> $log_file
}

debug_print()
{
    echo ""
    echo "$this_file: '${1}'"
    echo ""

    log_print "$1"
}

update_stats()
{
    [ "$enable_stat" != "1" ] && return

    [ ! -f $stat_file ] && {
	echo "exec_time \"start_date \"text_msg \"command \"result" > $stat_file
    }

    stat_total=`grep '$' -c $stat_file`

    [ $stat_total -gt $stat_max ] && return

    echo "$1 \"$start_date \"$2 \"$3 \"$4" >> $stat_file
}

#
# Parse command line arguments
#

while [ "$1" != "" ]
do
    case "$1" in
	-h)
	    shift;
	    usage
	    exit;;
	-c)
	    shift;
	    command="$1";;
	-t)
	    shift;
	    text_msg="$1";;
	-i)
	    shift;
	    more_msg="$1";;
	*)
	    usage
	    exit;;
    esac
    shift
done

[ "$command" = "" -o "$text_msg" = "" ] && {
    usage
    exit
}

#
# Boot time check
#

assumed_boot_time=60
current_uptime=`cat /proc/uptime | cut -d'.' -f1`
waiting_time=$(( assumed_boot_time - current_uptime ))

[ $waiting_time -gt 0 ] && {
    debug_print "Waiting for boot: $waiting_time seconds"
    sleep $waiting_time
}

start_date=$(date +%s)

#
# Execute command
#

debug_print "Executing: $command"

result=`eval "$command 2>>$log_file"`
ret="$?"

[ "$more_msg" != "" ] && result="$more_msg;answer:$result;"

#
# Print result (in other words, send result to the caller)
#

[ "$ret" = "0" ] && pass_or_fail="${text_msg}_pass" || pass_or_fail="${text_msg}_fail"
print_result "$pass_or_fail" "$result"

log_print "ret=$ret" "result=$result"

#
# Dump execution time in terms of seconds
#

end_date=$(date +%s)
dif_date=$(($end_date - $start_date))
debug_print "Consumed $dif_date seconds for ( $command )"

update_stats "$dif_date" "$pass_or_fail" "$command" "$result"
